<?php

namespace Tests\Feature;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Modules\Flowmaker\Models\Flow;
use Modules\Wpbox\Models\Contact;
use Modules\Wpbox\Models\Message;
use Modules\Wpbox\Events\ContactReplies;
use Modules\Flowmaker\Listeners\RespondOnMessage;

class FlowRoutingTest extends TestCase
{
    use RefreshDatabase;

    public function test_flow_routing_sets_current_flow()
    {
        // Create test contact
        $contact = Contact::factory()->create([
            'enabled_ai_bot' => true,
            'company_id' => 1
        ]);
        
        // Create test flow with keyword trigger
        $flow = Flow::factory()->create([
            'company_id' => 1,
            'priority' => 10,
            'is_active' => true,
            'flow_data' => json_encode([
                'nodes' => [
                    [
                        'type' => 'keyword_trigger',
                        'data' => [
                            'keywords' => [
                                ['value' => 'hosting', 'matchType' => 'contains']
                            ]
                        ]
                    ]
                ]
            ])
        ]);
        
        // Create test message
        $message = Message::factory()->create([
            'contact_id' => $contact->id,
            'value' => 'I need hosting help',
            'bot_has_replied' => false
        ]);
        
        // Create event
        $event = new ContactReplies($message);
        
        // Process the event
        $listener = new RespondOnMessage();
        $listener->handleMessageByContact($event);
        
        // Assert that current flow is set
        $this->assertEquals($flow->id, $contact->getCurrentFlowId());
    }

    public function test_flow_routing_continues_current_flow()
    {
        // Create test contact with current flow
        $contact = Contact::factory()->create([
            'enabled_ai_bot' => true,
            'company_id' => 1
        ]);
        
        // Create test flow
        $flow = Flow::factory()->create([
            'company_id' => 1,
            'priority' => 10,
            'is_active' => true
        ]);
        
        // Set current flow
        $contact->setCurrentFlowId($flow->id);
        
        // Create test message
        $message = Message::factory()->create([
            'contact_id' => $contact->id,
            'value' => 'Continue with current flow',
            'bot_has_replied' => false
        ]);
        
        // Create event
        $event = new ContactReplies($message);
        
        // Process the event
        $listener = new RespondOnMessage();
        $listener->handleMessageByContact($event);
        
        // Assert that current flow remains the same
        $this->assertEquals($flow->id, $contact->getCurrentFlowId());
    }

    public function test_flow_routing_falls_back_to_general_flow()
    {
        // Create test contact
        $contact = Contact::factory()->create([
            'enabled_ai_bot' => true,
            'company_id' => 1
        ]);
        
        // Create general flow (priority 0)
        $generalFlow = Flow::factory()->create([
            'company_id' => 1,
            'priority' => 0,
            'is_active' => true
        ]);
        
        // Create test message with no keyword matches
        $message = Message::factory()->create([
            'contact_id' => $contact->id,
            'value' => 'Hello, how are you?',
            'bot_has_replied' => false
        ]);
        
        // Create event
        $event = new ContactReplies($message);
        
        // Process the event
        $listener = new RespondOnMessage();
        $listener->handleMessageByContact($event);
        
        // Assert that general flow is used (no specific flow set)
        $this->assertNull($contact->getCurrentFlowId());
    }
}
