<?php

namespace Modules\Flowmaker\Models\Nodes;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Modules\Flowmaker\Models\Contact;

class FlowHTTPNode extends Node
{
    
    public function process($message, $data)
    {
        Log::info('Processing message in HTTP node', ['message' => $message, 'data' => $data]);
        
        try {
            // Get HTTP settings from node data
            $httpSettings = $this->getDataAsArray()['settings']['http'] ?? [];
            
            Log::info('HTTP Settings', ['httpSettings' => $httpSettings]);

            // Find the contact
            $contactId = is_object($data) ? $data->contact_id : $data['contact_id'];
            $contact = Contact::find($contactId);
            
            if (!$contact) {
                Log::error('Contact not found', ['contactId' => $contactId]);
                return ['success' => false];
            }

            Log::info('Contact found', ['contact' => $contact->id]);

            // Extract settings
            $method = strtoupper($httpSettings['method'] ?? 'GET');
            $url = $httpSettings['url'] ?? '';
            $headers = $httpSettings['headers'] ?? [];
            $params = $httpSettings['params'] ?? [];
            $responseVar = $httpSettings['responseVar'] ?? '';

            // Transform URL with variables
            $url = $contact->changeVariables($url, $this->flow_id);
            Log::info('Transformed URL', ['url' => $url]);

            if (empty($url)) {
                Log::error('URL is empty after transformation');
                return ['success' => false];
            }

            // Prepare headers
            $requestHeaders = [];
            foreach ($headers as $header) {
                if (!empty($header['key']) && !empty($header['value'])) {
                    $key = $contact->changeVariables($header['key'], $this->flow_id);
                    $value = $contact->changeVariables($header['value'], $this->flow_id);
                    $requestHeaders[$key] = $value;
                }
            }

            // Handle Basic Authentication if enabled
            if (!empty($httpSettings['useBasicAuth']) && $httpSettings['useBasicAuth']) {
                $username = $httpSettings['basicAuthUsername'] ?? '';
                $password = $httpSettings['basicAuthPassword'] ?? '';
                
                if (!empty($username) && !empty($password)) {
                    // Interpolate variables in credentials if needed
                    $username = $contact->changeVariables($username, $this->flow_id);
                    $password = $contact->changeVariables($password, $this->flow_id);
                    
                    // Create Basic Auth header
                    $credentials = $username . ':' . $password;
                    $base64Credentials = base64_encode($credentials);
                    $requestHeaders['Authorization'] = 'Basic ' . $base64Credentials;
                    
                    Log::info('Basic Auth header added', [
                        'username' => $username,
                        'password_length' => strlen($password)
                    ]);
                }
            }

            // Prepare parameters/data
            $requestData = [];
            foreach ($params as $param) {
                if (!empty($param['key']) && !empty($param['value'])) {
                    $key = $contact->changeVariables($param['key'], $this->flow_id);
                    $value = $contact->changeVariables($param['value'], $this->flow_id);
                    $requestData[$key] = $value;
                }
            }

            Log::info('Making HTTP request', [
                'method' => $method,
                'url' => $url,
                'headers' => $requestHeaders,
                'data' => $requestData
            ]);

            // Make HTTP request based on method
            $response = null;
            switch ($method) {
                case 'GET':
                    $response = Http::withHeaders($requestHeaders)->get($url, $requestData);
                    break;
                case 'POST':
                    $response = Http::withHeaders($requestHeaders)->post($url, $requestData);
                    break;
                case 'PUT':
                    $response = Http::withHeaders($requestHeaders)->put($url, $requestData);
                    break;
                case 'DELETE':
                    $response = Http::withHeaders($requestHeaders)->delete($url, $requestData);
                    break;
                default:
                    Log::error('Unsupported HTTP method', ['method' => $method]);
                    return ['success' => false];
            }

            if ($response) {
                $responseBody = $response->body();
                $responseSize = strlen($responseBody);
                
                Log::info('HTTP Response received', [
                    'status' => $response->status(),
                    'bodyLength' => $responseSize
                ]);

                // Check if response is too large for Pusher (10KB limit)
                if ($responseSize > 8000) { // Leave some buffer
                    Log::warning('Response too large for Pusher, truncating', [
                        'original_size' => $responseSize,
                        'max_size' => 8000
                    ]);
                    
                    // Truncate the response body
                    $responseBody = substr($responseBody, 0, 8000);
                    $responseBody .= '... [Response truncated due to size limit]';
                }

                $responseData = [
                    'status' => $response->status(),
                    'body' => $responseBody,
                    'json' => $response->json(),
                    'headers' => $response->headers()
                ];

                // Log authentication errors specifically
                if ($response->status() === 401) {
                    Log::error('Authentication failed - check your WooCommerce API credentials', [
                        'response_body' => $responseBody,
                        'url' => $url
                    ]);
                }

                // Store response in variable if specified
                if (!empty($responseVar)) {
                    $contact->setContactState($this->flow_id, $responseVar, json_encode($responseData));
                    Log::info('Response stored in variable', ['variable' => $responseVar]);
                }

                // Store individual components for easier access
                if (!empty($responseVar)) {
                    $contact->setContactState($this->flow_id, $responseVar . '_status', $response->status());
                    $contact->setContactState($this->flow_id, $responseVar . '_body', $response->body());
                    if ($response->json()) {
                        $contact->setContactState($this->flow_id, $responseVar . '_json', json_encode($response->json()));
                        
                        // Format WooCommerce products for WhatsApp
                        $jsonData = $response->json();
                        if ($this->isWooCommerceProductsResponse($jsonData)) {
                            $formattedProducts = $this->formatWooCommerceProducts($jsonData);
                            $contact->setContactState($this->flow_id, $responseVar . '_formatted', $formattedProducts);
                        }
                    }
                }

            } else {
                Log::error('HTTP request failed - no response');
                return ['success' => false];
            }

        } catch (\Exception $e) {
            Log::error('Error processing HTTP request', ['error' => $e->getMessage()]);
            return ['success' => false];
        }

        // Continue flow to next node if one exists
        $nextNode = $this->getNextNodeId();
        if ($nextNode) {
            $nextNode->process($message, $data);
        }

        return ['success' => true];
    }

    protected function getNextNodeId($data = null)
    {
        // Get the first outgoing edge's target
        if (!empty($this->outgoingEdges)) {
            return $this->outgoingEdges[0]->getTarget();
        }
        return null;
    }

    /**
     * Check if response is WooCommerce products data
     */
    protected function isWooCommerceProductsResponse($jsonData)
    {
        if (!is_array($jsonData)) {
            return false;
        }
        
        // Check if it's an array of products (WooCommerce products endpoint)
        if (isset($jsonData[0]) && is_array($jsonData[0])) {
            $firstItem = $jsonData[0];
            return isset($firstItem['id']) && 
                   isset($firstItem['name']) && 
                   isset($firstItem['price']) &&
                   isset($firstItem['type']);
        }
        
        return false;
    }

    /**
     * Format WooCommerce products for WhatsApp display
     */
    protected function formatWooCommerceProducts($products)
    {
        if (empty($products)) {
            return "❌ No products found.";
        }

        $formatted = "🛒 *Our Latest Products:*\n\n";
        $maxProducts = min(3, count($products));
        for ($i = 0; $i < $maxProducts; $i++) {
            $p = $products[$i];
            $formatted .= ($i+1) . ". *" . $p['name'] . "*\n";
            $formatted .= "   💰 Price: $" . $p['price'] . "\n";
            if (!empty($p['short_description'])) {
                $desc = strip_tags($p['short_description']);
                $formatted .= "   📝 " . $desc . "\n";
            }
            $formatted .= ($p['stock_status'] === 'instock' ? "   ✅ In Stock\n" : "   ❌ Out of Stock\n");
            // if (!empty($p['permalink'])) {
                // $formatted .= "   [View Product](" . $p['permalink'] . ")\n";
            // }
            $formatted .= "\n";
        }
        if (count($products) > $maxProducts) {
            $formatted .= "...and more products available!";
        }
        
        dd($formatted , $p);
        return $formatted;
    }
}
