<?php

namespace Modules\Contacts\Services;

use Modules\Contacts\Models\Contact;
use Modules\Contacts\Models\CrmConnection;
use Modules\Contacts\Models\ContactCrmMapping;
use Modules\Contacts\Models\CrmSyncHistory;
use Modules\Contacts\Services\CrmIntegration\CrmPluginRegistry;
use Modules\Contacts\Services\DuplicateDetectionService;
use Modules\Contacts\Services\PhoneNormalizationService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class EnhancedCrmImportService
{
    protected $crmRegistry;
    protected $duplicateService;
    protected $phoneService;

    public function __construct(
        CrmPluginRegistry $crmRegistry,
        DuplicateDetectionService $duplicateService,
        PhoneNormalizationService $phoneService
    ) {
        $this->crmRegistry = $crmRegistry;
        $this->duplicateService = $duplicateService;
        $this->phoneService = $phoneService;
    }

    /**
     * Import contacts from CRM.
     *
     * @param int $crmConnectionId
     * @param array $options
     * @return CrmSyncHistory
     * @throws \Exception
     */
    public function importContacts(int $crmConnectionId, array $options = []): CrmSyncHistory
    {
        $crmConnection = CrmConnection::findOrFail($crmConnectionId);
        $connector = $this->crmRegistry->get($crmConnection->type);

        // Create sync history record
        $syncHistory = CrmSyncHistory::create([
            'crm_connection_id' => $crmConnectionId,
            'sync_type' => 'import',
            'status' => 'pending',
            'import_settings' => $options,
            'field_mapping' => $crmConnection->field_mapping,
            'started_at' => now(),
        ]);

        try {
            $syncHistory->update(['status' => 'in_progress']);

            $importedCount = 0;
            $failedCount = 0;
            $duplicateCount = 0;
            $after = null;

            do {
                // Fetch contacts from CRM
                try {
                    $batchSize = min($options['batch_size'] ?? 100, 100); // HubSpot limit is 100
                    
                    // Build connector options
                    $connectorOptions = [
                        'limit' => $batchSize,
                        'after' => $after,
                        'properties' => $options['properties'] ?? ['firstname', 'lastname', 'email', 'phone', 'company'],
                    ];
                    
                    // Only add contact_type for Odoo connections
                    if ($crmConnection->type === 'odoo' && isset($options['contact_type'])) {
                        $connectorOptions['contact_type'] = $options['contact_type'];
                    }
                    
                    $crmData = $connector->getContacts($crmConnection->credentials, $connectorOptions);
                } catch (\Exception $e) {
                    // Handle authentication and API errors specifically
                    if (strpos($e->getMessage(), 'Authentication credentials not found') !== false) {
                        $syncHistory->update([
                            'status' => 'failed',
                            'error_message' => 'Invalid CRM credentials. Please check your connection settings and try again.',
                            'completed_at' => now(),
                        ]);
                        throw new \Exception('Invalid CRM credentials. Please check your connection settings and try again.');
                    } elseif (strpos($e->getMessage(), 'Failed to fetch contacts') !== false) {
                        $syncHistory->update([
                            'status' => 'failed',
                            'error_message' => 'Failed to connect to CRM. Please check your connection and try again.',
                            'completed_at' => now(),
                        ]);
                        throw new \Exception('Failed to connect to CRM. Please check your connection and try again.');
                    } else {
                        $syncHistory->update([
                            'status' => 'failed',
                            'error_message' => $e->getMessage(),
                            'completed_at' => now(),
                        ]);
                        throw $e;
                    }
                }

                $contacts = $crmData['contacts'] ?? [];
                $hasMore = $crmData['pagination']['has_more'] ?? false;
                $after = $crmData['pagination']['after'] ?? null;

                foreach ($contacts as $crmContact) {
                    try {
                        Log::info('Processing CRM contact', [
                            'crm_contact_id' => $crmContact['id'] ?? 'unknown',
                            'properties' => $crmContact['properties'] ?? [],
                        ]);
                        
                        $result = $this->processCrmContact($crmConnection, $crmContact, $options);
                        
                        Log::info('CRM contact processed', [
                            'crm_contact_id' => $crmContact['id'] ?? 'unknown',
                            'status' => $result['status'],
                            'message' => $result['message'] ?? '',
                        ]);
                        
                        if ($result['status'] === 'imported') {
                            $importedCount++;
                        } elseif ($result['status'] === 'duplicate') {
                            $duplicateCount++;
                        } else {
                            $failedCount++;
                        }
                    } catch (\Exception $e) {
                        Log::error('Error processing CRM contact', [
                            'crm_contact_id' => $crmContact['id'] ?? 'unknown',
                            'error' => $e->getMessage(),
                            'trace' => $e->getTraceAsString(),
                        ]);
                        $failedCount++;
                    }
                }

                // Update progress
                $syncHistory->update([
                    'processed_records' => $syncHistory->processed_records + count($contacts),
                    'successful_records' => $importedCount,
                    'failed_records' => $failedCount,
                    'duplicate_records' => $duplicateCount,
                ]);

                // Update cache for real-time progress
                Cache::put("crm_import_progress_{$syncHistory->id}", [
                    'processed' => $syncHistory->processed_records,
                    'total' => $syncHistory->total_records,
                    'imported' => $importedCount,
                    'failed' => $failedCount,
                    'duplicates' => $duplicateCount,
                ], 3600);

            } while ($hasMore);

            // Update final status
            $syncHistory->update([
                'status' => 'completed',
                'completed_at' => now(),
                'total_records' => $syncHistory->processed_records,
            ]);

            // Update CRM connection last sync
            $crmConnection->update(['last_sync_at' => now()]);

            return $syncHistory;

        } catch (\Exception $e) {
            Log::error('CRM import failed', [
                'crm_connection_id' => $crmConnectionId,
                'sync_history_id' => $syncHistory->id,
                'error' => $e->getMessage(),
            ]);

            $syncHistory->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
                'completed_at' => now(),
            ]);

            throw $e;
        }
    }

    /**
     * Process a single CRM contact.
     *
     * @param CrmConnection $crmConnection
     * @param array $crmContact
     * @param array $options
     * @return array
     */
    protected function processCrmContact(CrmConnection $crmConnection, array $crmContact, array $options): array
    {
        try {
            // Transform CRM contact data to our format
            $contactData = $this->transformCrmContactData($crmContact, $crmConnection->field_mapping);

            // Validate contact data before processing
            $validationResult = $this->validateContactData($contactData);
            if (!$validationResult['valid']) {
                return ['status' => 'failed', 'message' => $validationResult['message']];
            }

            // Check for duplicates
            $duplicateResult = $this->duplicateService->findDuplicates($contactData);
            
            Log::info('Duplicate check result', [
                'contact_data' => $contactData,
                'has_duplicates' => $duplicateResult['has_duplicates'],
                'duplicate_count' => $duplicateResult['duplicate_count'],
                'duplicate_strategy' => $options['duplicate_strategy'],
            ]);

            if ($duplicateResult['has_duplicates']) {
                if ($options['duplicate_strategy'] === 'skip') {
                    return ['status' => 'duplicate', 'message' => 'Contact skipped due to duplicate'];
                }

                if ($options['duplicate_strategy'] === 'update') {
                    $contact = $duplicateResult['duplicates']->first();
                    $this->updateContactFromCrm($contact, $contactData, $crmConnection, $crmContact);
                    return ['status' => 'imported', 'message' => 'Contact updated'];
                }

                if ($options['duplicate_strategy'] === 'merge') {
                    $contact = $duplicateResult['duplicates']->first();
                    $this->mergeContactFromCrm($contact, $contactData, $crmConnection, $crmContact);
                    return ['status' => 'imported', 'message' => 'Contact merged'];
                }
            }

            // Create new contact
            $contact = $this->createContactFromCrm($contactData, $crmConnection, $crmContact);

            return ['status' => 'imported', 'message' => 'Contact created'];
            
        } catch (\Exception $e) {
            Log::error('Error in processCrmContact', [
                'crm_contact_id' => $crmContact['id'] ?? 'unknown',
                'error' => $e->getMessage(),
                'contact_data' => $contactData ?? null,
            ]);
            return ['status' => 'failed', 'message' => 'Error processing contact: ' . $e->getMessage()];
        }
    }

    /**
     * Validate contact data before processing.
     *
     * @param array $contactData
     * @return array
     */
    protected function validateContactData(array $contactData): array
    {
        // Check if we have at least a name
        if (empty($contactData['name']) || trim($contactData['name']) === '') {
            return ['valid' => false, 'message' => 'Contact name is required'];
        }

        // Check if we have at least email or phone
        if (empty($contactData['email']) && empty($contactData['phone'])) {
            return ['valid' => false, 'message' => 'Contact must have either email or phone'];
        }

        // Validate email format if present
        if (!empty($contactData['email']) && !filter_var($contactData['email'], FILTER_VALIDATE_EMAIL)) {
            return ['valid' => false, 'message' => 'Invalid email format'];
        }

        return ['valid' => true, 'message' => 'Contact data is valid'];
    }

    /**
     * Transform CRM contact data to our format.
     *
     * @param array $crmContact
     * @param array $fieldMapping
     * @return array
     */
    protected function transformCrmContactData(array $crmContact, array $fieldMapping): array
    {
        // Detect CRM format - HubSpot has 'properties' wrapper, others don't
        $properties = isset($crmContact['properties']) ? $crmContact['properties'] : $crmContact;
        
        // Get name components with CRM-specific field mapping
        $firstName = $this->getMappedValue($properties, $fieldMapping['name'] ?? 'firstname') ?? 
                    $this->getMappedValue($properties, 'FirstName') ?? // Salesforce
                    $this->getMappedValue($properties, 'first_name') ?? // Pipedrive
                    '';
        
        $lastName = $this->getMappedValue($properties, $fieldMapping['lastname'] ?? 'lastname') ?? 
                   $this->getMappedValue($properties, 'LastName') ?? // Salesforce
                   $this->getMappedValue($properties, 'last_name') ?? // Pipedrive
                   '';
        
        // Build full name with smart fallbacks
        $fullName = trim($firstName . ' ' . $lastName);
        if (empty($fullName)) {
            // Try alternative name fields for different CRMs
            $fullName = $this->getMappedValue($properties, 'name') ?? 
                       $this->getMappedValue($properties, 'full_name') ?? 
                       $this->getMappedValue($properties, 'display_name') ?? 
                       $this->getMappedValue($properties, 'Name') ?? // Salesforce
                       'Unknown Contact';
        }
        
        // Get email with CRM-specific fallbacks
        $email = $this->getMappedValue($properties, $fieldMapping['email'] ?? 'email') ?? 
                $this->getMappedValue($properties, 'Email') ?? // Salesforce
                $this->getMappedValue($properties, 'email_address') ?? 
                $this->getMappedValue($properties, 'work_email') ?? 
                null;
        
        // Get phone with CRM-specific fallbacks
        $phone = $this->getMappedValue($properties, $fieldMapping['phone'] ?? 'phone') ?? 
                $this->getMappedValue($properties, 'Phone') ?? // Salesforce
                $this->getMappedValue($properties, 'mobile_phone') ?? 
                $this->getMappedValue($properties, 'work_phone') ?? 
                $this->getMappedValue($properties, 'phone_number') ?? 
                $this->getMappedValue($properties, 'mobile') ?? // Odoo uses 'mobile' field
                $this->getMappedValue($properties, 'phone') ?? // Pipedrive
                null;
        
        $contactData = [
            'name' => $fullName,
            'email' => $email,
            'phone' => $phone,
        ];

        // Normalize phone number if present
        if ($contactData['phone']) {
            $contactData['phone'] = $this->phoneService->normalizePhone($contactData['phone']);
        }

        // Ensure we have at least email or phone for contact identification
        if (empty($contactData['email']) && empty($contactData['phone'])) {
            // Generate a placeholder email if neither email nor phone is available
            $contactData['email'] = 'no-email-' . ($crmContact['id'] ?? $crmContact['Id'] ?? uniqid()) . '@imported.local';
        }

        return $contactData;
    }

    /**
     * Get mapped value from CRM properties.
     *
     * @param array $properties
     * @param string $fieldName
     * @return string|null
     */
    protected function getMappedValue(array $properties, string $fieldName): ?string
    {
        return $properties[$fieldName] ?? null;
    }

    /**
     * Create a new contact from CRM data.
     *
     * @param array $contactData
     * @param CrmConnection $crmConnection
     * @param array $crmContact
     * @return Contact
     */
    protected function createContactFromCrm(array $contactData, CrmConnection $crmConnection, array $crmContact): Contact
    {
        return DB::transaction(function () use ($contactData, $crmConnection, $crmContact) {
            // Create contact
            $contact = Contact::create($contactData);

            // Create CRM mapping
            ContactCrmMapping::create([
                'contact_id' => $contact->id,
                'crm_connection_id' => $crmConnection->id,
                'crm_contact_id' => $crmContact['id'],
                'sync_status' => 'synced',
                'crm_data' => $crmContact,
                'last_synced_at' => now(),
            ]);

            return $contact;
        });
    }

    /**
     * Update existing contact from CRM data.
     *
     * @param Contact $contact
     * @param array $contactData
     * @param CrmConnection $crmConnection
     * @param array $crmContact
     * @return void
     */
    protected function updateContactFromCrm(Contact $contact, array $contactData, CrmConnection $crmConnection, array $crmContact): void
    {
        DB::transaction(function () use ($contact, $contactData, $crmConnection, $crmContact) {
            // Update contact
            $contact->update($contactData);

            // Update or create CRM mapping
            ContactCrmMapping::updateOrCreate(
                [
                    'contact_id' => $contact->id,
                    'crm_connection_id' => $crmConnection->id,
                ],
                [
                    'crm_contact_id' => $crmContact['id'],
                    'sync_status' => 'synced',
                    'crm_data' => $crmContact,
                    'last_synced_at' => now(),
                ]
            );
        });
    }

    /**
     * Merge existing contact with CRM data.
     *
     * @param Contact $contact
     * @param array $contactData
     * @param CrmConnection $crmConnection
     * @param array $crmContact
     * @return void
     */
    protected function mergeContactFromCrm(Contact $contact, array $contactData, CrmConnection $crmConnection, array $crmContact): void
    {
        DB::transaction(function () use ($contact, $contactData, $crmConnection, $crmContact) {
            // Merge data (keep existing values if new ones are empty)
            $mergedData = array_merge($contact->toArray(), array_filter($contactData));
            $contact->update($mergedData);

            // Update or create CRM mapping
            ContactCrmMapping::updateOrCreate(
                [
                    'contact_id' => $contact->id,
                    'crm_connection_id' => $crmConnection->id,
                ],
                [
                    'crm_contact_id' => $crmContact['id'],
                    'sync_status' => 'synced',
                    'crm_data' => $crmContact,
                    'last_synced_at' => now(),
                ]
            );
        });
    }

    /**
     * Get import progress.
     *
     * @param int $syncHistoryId
     * @return array
     */
    public function getImportProgress(int $syncHistoryId): array
    {
        $syncHistory = CrmSyncHistory::findOrFail($syncHistoryId);
        
        return [
            'status' => $syncHistory->status,
            'processed' => $syncHistory->processed_records,
            'total' => $syncHistory->total_records,
            'imported' => $syncHistory->successful_records,
            'failed' => $syncHistory->failed_records,
            'duplicates' => $syncHistory->duplicate_records,
            'progress_percentage' => $syncHistory->progress_percentage,
            'started_at' => $syncHistory->started_at,
            'completed_at' => $syncHistory->completed_at,
        ];
    }

    /**
     * Cancel an import.
     *
     * @param int $syncHistoryId
     * @return bool
     */
    public function cancelImport(int $syncHistoryId): bool
    {
        $syncHistory = CrmSyncHistory::findOrFail($syncHistoryId);
        
        if ($syncHistory->status === 'in_progress') {
            $syncHistory->update([
                'status' => 'cancelled',
                'completed_at' => now(),
            ]);
            
            return true;
        }
        
        return false;
    }
}