<?php

namespace Modules\Contacts\Services;

use Modules\Contacts\Models\Contact;
use Illuminate\Support\Collection;

class DuplicateDetectionService
{
    protected $phoneService;

    public function __construct(PhoneNormalizationService $phoneService)
    {
        $this->phoneService = $phoneService;
    }

    /**
     * Find duplicate contacts.
     *
     * @param array $contactData
     * @return array
     */
    public function findDuplicates(array $contactData): array
    {
        $duplicates = collect();

        // Check by phone number
        if (!empty($contactData['phone'])) {
            $phoneDuplicates = $this->findByPhone($contactData['phone']);
            $duplicates = $duplicates->merge($phoneDuplicates);
        }

        // Check by email
        if (!empty($contactData['email'])) {
            $emailDuplicates = $this->findByEmail($contactData['email']);
            $duplicates = $duplicates->merge($emailDuplicates);
        }

        // Check by name similarity
        if (!empty($contactData['name'])) {
            $nameDuplicates = $this->findByNameSimilarity($contactData['name']);
            $duplicates = $duplicates->merge($nameDuplicates);
        }

        return [
            'has_duplicates' => $duplicates->isNotEmpty(),
            'duplicates' => $duplicates->unique('id'),
            'duplicate_count' => $duplicates->unique('id')->count(),
        ];
    }

    /**
     * Find duplicates by phone number.
     *
     * @param string $phone
     * @return Collection
     */
    protected function findByPhone(string $phone): Collection
    {
        $variations = $this->phoneService->getVariations($phone);
        
        return Contact::where(function ($query) use ($variations) {
            foreach ($variations as $variation) {
                $query->orWhere('phone', $variation);
            }
        })->get();
    }

    /**
     * Find duplicates by email.
     *
     * @param string $email
     * @return Collection
     */
    protected function findByEmail(string $email): Collection
    {
        return Contact::where('email', $email)->get();
    }

    /**
     * Find duplicates by name similarity.
     *
     * @param string $name
     * @return Collection
     */
    protected function findByNameSimilarity(string $name): Collection
    {
        $similarityThreshold = 0.8; // 80% similarity
        $contacts = Contact::all();
        
        return $contacts->filter(function ($contact) use ($name, $similarityThreshold) {
            $similarity = $this->calculateNameSimilarity($name, $contact->name);
            return $similarity >= $similarityThreshold;
        });
    }

    /**
     * Calculate name similarity using Levenshtein distance.
     *
     * @param string $name1
     * @param string $name2
     * @return float
     */
    protected function calculateNameSimilarity(string $name1, string $name2): float
    {
        $name1 = strtolower(trim($name1));
        $name2 = strtolower(trim($name2));

        if ($name1 === $name2) {
            return 1.0;
        }

        $maxLength = max(strlen($name1), strlen($name2));
        if ($maxLength === 0) {
            return 0.0;
        }

        $distance = levenshtein($name1, $name2);
        return 1 - ($distance / $maxLength);
    }

    /**
     * Get duplicate detection strategies.
     *
     * @return array
     */
    public function getStrategies(): array
    {
        return [
            'skip' => [
                'label' => 'Skip Duplicates',
                'description' => 'Skip contacts that already exist',
            ],
            'update' => [
                'label' => 'Update Existing',
                'description' => 'Update existing contacts with new data',
            ],
            'merge' => [
                'label' => 'Merge Data',
                'description' => 'Merge new data with existing contact data',
            ],
        ];
    }

    /**
     * Get duplicate detection criteria.
     *
     * @return array
     */
    public function getCriteria(): array
    {
        return [
            'phone' => [
                'label' => 'Phone Number',
                'description' => 'Match by phone number (with normalization)',
                'weight' => 1.0,
            ],
            'email' => [
                'label' => 'Email Address',
                'description' => 'Match by exact email address',
                'weight' => 0.9,
            ],
            'name' => [
                'label' => 'Name Similarity',
                'description' => 'Match by name similarity (80% threshold)',
                'weight' => 0.7,
            ],
        ];
    }
}