<?php

namespace Modules\Contacts\Services\CrmIntegration\Connectors;

use Modules\Contacts\Services\CrmIntegration\CrmConnectorInterface;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

class SalesforceConnector implements CrmConnectorInterface
{
    public function __construct()
    {
        // No credentials stored in constructor
    }

    /**
     * Test the CRM connection
     */
    public function testConnection(array $credentials): bool
    {
        try {
            $baseUrl = rtrim($credentials['instance_url'], '/') . '/services/data/v58.0';
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json'
            ])->get($baseUrl . '/sobjects/Contact/describe');

            return $response->successful();

        } catch (Exception $e) {
            Log::error('Salesforce connection test failed', [
                'error' => $e->getMessage(),
                'credentials' => $credentials
            ]);

            return false;
        }
    }

    /**
     * Get contacts from Salesforce
     */
    public function getContacts(array $credentials, array $options = []): array
    {
        try {
            $baseUrl = rtrim($credentials['instance_url'], '/') . '/services/data/v58.0';
            $limit = min($options['limit'] ?? 100, 200); // Salesforce limit
            $offset = $options['offset'] ?? 0;

            $query = "SELECT Id, FirstName, LastName, Email, Phone, Company, CreatedDate, LastModifiedDate 
                     FROM Contact 
                     ORDER BY LastModifiedDate DESC 
                     LIMIT {$limit} 
                     OFFSET {$offset}";

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json'
            ])->get($baseUrl . '/query', [
                'q' => $query
            ]);

            if ($response->successful()) {
                $data = $response->json();
                return [
                    'success' => true,
                    'contacts' => $data['records'] ?? [],
                    'total' => $data['totalSize'] ?? 0,
                    'has_more' => count($data['records'] ?? []) >= $limit
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to fetch Salesforce contacts: ' . $response->body()
            ];

        } catch (Exception $e) {
            Log::error('Salesforce contacts fetch failed', [
                'error' => $e->getMessage(),
                'options' => $options
            ]);

            return [
                'success' => false,
                'message' => 'Failed to fetch Salesforce contacts: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get a single contact from Salesforce
     */
    public function getContact(array $credentials, string $contactId): ?array
    {
        try {
            $baseUrl = rtrim($credentials['instance_url'], '/') . '/services/data/v58.0';
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json'
            ])->get($baseUrl . '/sobjects/Contact/' . $contactId);

            if ($response->successful()) {
                return $response->json();
            }

            return null;

        } catch (Exception $e) {
            Log::error('Salesforce contact fetch failed', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId
            ]);

            return null;
        }
    }

    /**
     * Create a contact in Salesforce
     */
    public function createContact(array $credentials, array $contactData): array
    {
        try {
            $baseUrl = rtrim($credentials['instance_url'], '/') . '/services/data/v58.0';
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json'
            ])->post($baseUrl . '/sobjects/Contact', $contactData);

            if ($response->successful()) {
                return $response->json();
            }

            throw new Exception('Failed to create Salesforce contact: ' . $response->body());

        } catch (Exception $e) {
            Log::error('Salesforce contact creation failed', [
                'error' => $e->getMessage(),
                'contact_data' => $contactData
            ]);

            throw $e;
        }
    }

    /**
     * Update a contact in Salesforce
     */
    public function updateContact(array $credentials, string $contactId, array $contactData): array
    {
        try {
            $baseUrl = rtrim($credentials['instance_url'], '/') . '/services/data/v58.0';
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json'
            ])->patch($baseUrl . '/sobjects/Contact/' . $contactId, $contactData);

            if ($response->successful()) {
                return $response->json();
            }

            throw new Exception('Failed to update Salesforce contact: ' . $response->body());

        } catch (Exception $e) {
            Log::error('Salesforce contact update failed', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId,
                'contact_data' => $contactData
            ]);

            throw $e;
        }
    }

    /**
     * Delete a contact from Salesforce
     */
    public function deleteContact(array $credentials, string $contactId): bool
    {
        try {
            $baseUrl = rtrim($credentials['instance_url'], '/') . '/services/data/v58.0';
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json'
            ])->delete($baseUrl . '/sobjects/Contact/' . $contactId);

            return $response->successful();

        } catch (Exception $e) {
            Log::error('Salesforce contact deletion failed', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId
            ]);

            return false;
        }
    }

    /**
     * Get available fields from Salesforce
     */
    public function getAvailableFields(array $credentials): array
    {
        try {
            $baseUrl = rtrim($credentials['instance_url'], '/') . '/services/data/v58.0';
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json'
            ])->get($baseUrl . '/sobjects/Contact/describe');

            if ($response->successful()) {
                $data = $response->json();
                $fields = [];

                // Define essential fields for contact mapping
                $essentialFields = [
                    'FirstName', 'LastName', 'Email', 'Phone', 'MobilePhone', 'Company',
                    'Title', 'Department', 'Industry', 'Website', 'LeadSource',
                    'MailingAddress', 'MailingCity', 'MailingState', 'MailingCountry', 'MailingPostalCode',
                    'OtherAddress', 'OtherCity', 'OtherState', 'OtherCountry', 'OtherPostalCode',
                    'CreatedDate', 'LastModifiedDate', 'LastActivityDate', 'LastContactedDate',
                    'Description', 'Notes__c', 'Status__c'
                ];

                foreach ($data['fields'] ?? [] as $field) {
                    // Only include essential fields to avoid overwhelming the dropdown
                    if (in_array($field['name'], $essentialFields)) {
                        $fields[] = [
                            'name' => $field['name'],
                            'label' => $field['label'],
                            'type' => $field['type'],
                            'required' => $field['nillable'] === false,
                            'updateable' => $field['updateable'] ?? false
                        ];
                    }
                }

                return $fields;
            }

            throw new Exception('Failed to fetch Salesforce fields: ' . $response->body());

        } catch (Exception $e) {
            Log::error('Salesforce fields fetch failed', [
                'error' => $e->getMessage()
            ]);

            throw $e;
        }
    }

    /**
     * Transform Salesforce contact data to our system format
     */
    public function transformContactData(array $crmContact): array
    {
        return [
            'name' => trim(($crmContact['FirstName'] ?? '') . ' ' . ($crmContact['LastName'] ?? '')),
            'email' => $crmContact['Email'] ?? null,
            'phone' => $crmContact['Phone'] ?? null,
            'company' => $crmContact['Company'] ?? null,
            'crm_id' => $crmContact['Id'] ?? null,
            'crm_data' => $crmContact,
            'created_at' => $crmContact['CreatedDate'] ?? now(),
            'updated_at' => $crmContact['LastModifiedDate'] ?? now()
        ];
    }

    /**
     * Get CRM type
     */
    public function getType(): string
    {
        return 'salesforce';
    }

    /**
     * Get CRM display name
     */
    public function getDisplayName(): string
    {
        return 'Salesforce';
    }

    /**
     * Get required credentials
     */
    public function getRequiredCredentials(): array
    {
        return [
            'access_token' => 'Access Token',
            'instance_url' => 'Instance URL (e.g., https://your-instance.salesforce.com)',
            'refresh_token' => 'Refresh Token (optional)',
            'client_id' => 'Client ID',
            'client_secret' => 'Client Secret'
        ];
    }

    /**
     * Validate credentials
     */
    public function validateCredentials(array $credentials): bool
    {
        $required = ['access_token', 'instance_url', 'client_id', 'client_secret'];
        
        foreach ($required as $field) {
            if (empty($credentials[$field])) {
                return false;
            }
        }

        return true;
    }
}