<?php

namespace Modules\Contacts\Models;

use App\Models\Company;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Scopes\CompanyScope;
use Illuminate\Database\Eloquent\Casts\Attribute;

class CrmConnection extends Model
{
    use SoftDeletes;

    protected $table = 'crm_connections';
    public $guarded = [];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'credentials' => 'encrypted:array',
        'field_mapping' => 'array',
        'is_active' => 'boolean',
        'is_healthy' => 'boolean',
        'last_sync_at' => 'datetime',
        'last_test_at' => 'datetime',
    ];

    /**
     * Boot the model.
     */
    protected static function booted()
    {
        static::addGlobalScope(new CompanyScope);

        static::creating(function ($model) {
            $company_id = session('company_id', null);
            
            // If no company_id in session, try to get it from the authenticated user
            if (!$company_id && auth()->check() && auth()->user()->company_id) {
                $company_id = auth()->user()->company_id;
            }
            
            // If still no company_id, try to get the first company (for testing purposes)
            if (!$company_id) {
                $company = \App\Models\Company::first();
                if ($company) {
                    $company_id = $company->id;
                }
            }
            
            if ($company_id) {
                $model->company_id = $company_id;
            }
        });
    }

    /**
     * Get the company that owns the CRM connection.
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the contact mappings for this CRM connection.
     */
    public function contactMappings()
    {
        return $this->hasMany(ContactCrmMapping::class);
    }

    /**
     * Get the sync history for this CRM connection.
     */
    public function syncHistory()
    {
        return $this->hasMany(CrmSyncHistory::class);
    }

    /**
     * Get the contacts through the mapping.
     */
    public function contacts()
    {
        return $this->belongsToMany(
            Contact::class,
            'contact_crm_mapping',
            'crm_connection_id',
            'contact_id'
        )->withPivot(['crm_contact_id', 'sync_status', 'crm_data', 'last_synced_at', 'sync_error']);
    }

    /**
     * Get the latest sync history.
     */
    public function latestSync()
    {
        return $this->hasOne(CrmSyncHistory::class)->latest();
    }

    /**
     * Check if the connection is healthy.
     */
    public function isHealthy(): bool
    {
        return $this->is_healthy && $this->is_active;
    }

    /**
     * Get the connection status.
     */
    public function getStatusAttribute(): string
    {
        if (!$this->is_active) {
            return 'inactive';
        }

        if (!$this->is_healthy) {
            return 'unhealthy';
        }

        if ($this->last_test_at && $this->last_test_at->diffInHours(now()) > 24) {
            return 'stale';
        }

        return 'healthy';
    }

    /**
     * Get the CRM type display name.
     */
    public function getTypeDisplayAttribute(): string
    {
        return ucfirst($this->type);
    }

    /**
     * Scope to get only active connections.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to get only healthy connections.
     */
    public function scopeHealthy($query)
    {
        return $query->where('is_healthy', true);
    }

    /**
     * Scope to get connections by type.
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }
}