import { useMemo } from 'react';
import { useReactFlow } from '@xyflow/react';
import { variables } from '@/data/variablesData';

type Variable = {
  label: string;
  value: string;
  category: string;
};

export function useFlowVariables() {
  const { getNodes } = useReactFlow();

  // Extract variables from flow nodes
  const flowVariables = useMemo(() => {
    const nodes = getNodes();
    const dynamicVariables: Variable[] = [];

    nodes.forEach(node => {
      const nodeData = node.data as any;
      const settings = nodeData?.settings || {};
      
      if (node.type === 'openai' && settings.llm?.variableName) {
        dynamicVariables.push({
          label: `Full AI Response for ${settings.llm.variableName}`,
          value: settings.llm.variableName,
          category: 'Flow Variables'
        },{
          label: `AI Response  for ${settings.llm.variableName}`,
          value: settings.llm.variableName + '_message',
          category: 'Flow Variables'
        },{
          label: `Detected Intent for ${settings.llm.variableName}`,
          value: settings.llm.variableName + '_intent',
          category: 'Flow Variables'
        });
      }
      
      if (node.type === 'question' && settings.variableName) {
        dynamicVariables.push({
          label: `User Answer (${nodeData.label || 'Question'})`,
          value: settings.variableName,
          category: 'Flow Variables'
        });
      }
      
      if (node.type === 'http' && settings.http?.responseVar) {
        const base = settings.http.responseVar as string;
        const nodeLabel = nodeData.label || 'HTTP Request';
        dynamicVariables.push(
          { label: `HTTP Response (${nodeLabel})`, value: base, category: 'Flow Variables' },
          { label: `HTTP Status (${nodeLabel})`, value: `${base}.status`, category: 'Flow Variables' },
          { label: `HTTP Data (${nodeLabel})`, value: `${base}.data`, category: 'Flow Variables' },
          { label: `HTTP Headers (${nodeLabel})`, value: `${base}.headers`, category: 'Flow Variables' },
          // Runtime (saved) variants
          { label: `HTTP Status (saved) (${nodeLabel})`, value: `${base}_status`, category: 'Flow Variables' },
          { label: `HTTP Data (saved) (${nodeLabel})`, value: `${base}_data`, category: 'Flow Variables' },
          { label: `HTTP Headers (saved) (${nodeLabel})`, value: `${base}_headers`, category: 'Flow Variables' },
          { label: `HTTP JSON (saved) (${nodeLabel})`, value: `${base}_json`, category: 'Flow Variables' },
          { label: `HTTP Total (saved) (${nodeLabel})`, value: `${base}_total`, category: 'Flow Variables' },
          { label: `HTTP Has Results (saved) (${nodeLabel})`, value: `${base}_has_results`, category: 'Flow Variables' },
          { label: `HTTP Results Count (saved) (${nodeLabel})`, value: `${base}_results_count`, category: 'Flow Variables' },
          // JSON alias for convenience
          { label: `HTTP JSON (${nodeLabel})`, value: `${base}.json`, category: 'Flow Variables' }
        );
      }

      // Support Advanced HTTP node (new_request)
      if (node.type === 'new_request') {
        const base: string | undefined = settings.http?.responseVar || settings.new_request?.responseVar;
        const nodeLabel = nodeData.label || 'Advanced HTTP Request';
        if (base) {
          dynamicVariables.push(
            { label: `HTTP Response (${nodeLabel})`, value: base, category: 'Flow Variables' },
            { label: `HTTP Status (${nodeLabel})`, value: `${base}.status`, category: 'Flow Variables' },
            { label: `HTTP Data (${nodeLabel})`, value: `${base}.data`, category: 'Flow Variables' },
            { label: `HTTP Headers (${nodeLabel})`, value: `${base}.headers`, category: 'Flow Variables' },
            // Runtime (saved) variants
            { label: `HTTP Status (saved) (${nodeLabel})`, value: `${base}_status`, category: 'Flow Variables' },
            { label: `HTTP Data (saved) (${nodeLabel})`, value: `${base}_data`, category: 'Flow Variables' },
            { label: `HTTP Headers (saved) (${nodeLabel})`, value: `${base}_headers`, category: 'Flow Variables' },
            { label: `HTTP JSON (saved) (${nodeLabel})`, value: `${base}_json`, category: 'Flow Variables' },
            { label: `HTTP Total (saved) (${nodeLabel})`, value: `${base}_total`, category: 'Flow Variables' },
            { label: `HTTP Has Results (saved) (${nodeLabel})`, value: `${base}_has_results`, category: 'Flow Variables' },
            { label: `HTTP Results Count (saved) (${nodeLabel})`, value: `${base}_results_count`, category: 'Flow Variables' },
            // JSON alias for convenience
            { label: `HTTP JSON (${nodeLabel})`, value: `${base}.json`, category: 'Flow Variables' }
          );
        }
      }
      
      if (node.type === 'datastore' && settings.variableName) {
        dynamicVariables.push({
          label: `Data Store (${nodeData.label || 'Data Store'})`,
          value: settings.variableName,
          category: 'Flow Variables'
        });
      }
    });

    return dynamicVariables;
  }, [getNodes]);

  // Combine static and dynamic variables
  const allVariables = useMemo(() => {
    return [...variables, ...flowVariables];
  }, [flowVariables]);

  // Group variables by category
  const groupedVariables = useMemo(() => {
    return allVariables.reduce((acc, variable) => {
      const category = variable.category || 'Other';
      if (!acc[category]) {
        acc[category] = [];
      }
      acc[category].push(variable);
      return acc;
    }, {} as Record<string, Variable[]>);
  }, [allVariables]);

  return {
    allVariables,
    groupedVariables,
    flowVariables
  };
} 