<?php

namespace Modules\Flowmaker\Models\Nodes;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Modules\Flowmaker\Models\Contact;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Mail\Mailables\Attachment;
use Symfony\Component\Mailer\Transport\Smtp\EsmtpTransport;

class SendEmail extends Node
{
    public function process($message, $data)
    {
        Log::info('🔄 Processing SendEmail node', [
            'node_id' => $this->id ?? 'unknown',
            'flow_id' => $this->flow_id ?? 'unknown'
        ]);

        try {
            // Load settings
            $nodeData = $this->getDataAsArray();
            $settings = $nodeData['settings'] ?? [];
            Log::debug('📋 SendEmail settings', ['settings' => $settings]);

            // Resolve contact
            $contactId = is_object($data) ? ($data->contact_id ?? null) : ($data['contact_id'] ?? null);
            $contact = $contactId ? Contact::find($contactId) : null;
            if (!$contact) {
                Log::error('❌ Contact not found', ['contactId' => $contactId]);
                return ['success' => false];
            }

            // Extract settings - Initialize all variables first
            $to = '';
            $cc = '';
            $bcc = '';
            $subject = '';
            $body = '';
            $isHtml = true; // Default to HTML
            $fromEmail = '';
            $fromName = '';
            $responseVar = '';
            $attachments = [];
            $smtpConfig = [];
            
            // Now assign values
            if (isset($settings['to'])) {
                $to = $settings['to'];
            }
            if (isset($settings['cc'])) {
                $cc = $settings['cc'];
            }
            if (isset($settings['bcc'])) {
                $bcc = $settings['bcc'];
            }
            if (isset($settings['subject'])) {
                $subject = $settings['subject'];
            }
            if (isset($settings['body'])) {
                $body = $settings['body'];
            }
            if (isset($settings['isHtml'])) {
                $isHtml = (bool) $settings['isHtml'];
            }
            if (isset($settings['fromEmail'])) {
                $fromEmail = $settings['fromEmail'];
            }
            if (isset($settings['fromName'])) {
                $fromName = $settings['fromName'];
            }
            if (isset($settings['responseVar'])) {
                $responseVar = trim($settings['responseVar']);
            }
            if (isset($settings['attachments'])) {
                $attachments = $settings['attachments'];
            }
            if (isset($settings['smtpConfig'])) {
                $smtpConfig = $settings['smtpConfig'];
            }

            // Safety check - ensure all variables are defined
            if (!isset($to)) {
                Log::error('❌ Variable $to is not defined after initialization');
                return ['success' => false, 'error' => 'Variable initialization failed'];
            }
            
            // Debug: Log initial settings
            Log::debug('📧 Initial email settings', [
                'to' => $to,
                'subject' => $subject,
                'fromEmail' => $fromEmail,
                'settings_keys' => array_keys($settings),
                'all_variables_defined' => [
                    'to' => isset($to),
                    'cc' => isset($cc),
                    'bcc' => isset($bcc),
                    'subject' => isset($subject),
                    'body' => isset($body),
                    'fromEmail' => isset($fromEmail),
                    'fromName' => isset($fromName)
                ]
            ]);

            // Interpolate variables
            $originalTo = $to;
            $to = $contact->changeVariables($to, $this->flow_id);
            $cc = $contact->changeVariables($cc, $this->flow_id);
            $bcc = $contact->changeVariables($bcc, $this->flow_id);
            $subject = $contact->changeVariables($subject, $this->flow_id);
            $body = $contact->changeVariables($body, $this->flow_id);
            $fromEmail = $contact->changeVariables($fromEmail, $this->flow_id);
            $fromName = $contact->changeVariables($fromName, $this->flow_id);

            Log::debug('📧 Email variables after interpolation', [
                'original_to' => $originalTo,
                'interpolated_to' => $to,
                'subject' => $subject,
                'fromEmail' => $fromEmail,
                'fromName' => $fromName
            ]);

            if (empty($to)) {
                Log::error('❌ Recipient email is empty', [
                    'original_to' => $originalTo,
                    'interpolated_to' => $to,
                    'settings' => $settings
                ]);
                return ['success' => false, 'error' => 'Recipient email is required'];
            }

            // Create custom mailable
            $mailable = new class($subject, $body, $isHtml, $fromEmail, $fromName, $attachments) extends Mailable {
                public $subject;
                public $body;
                public $isHtml;
                public $fromEmail;
                public $fromName;
                public $attachments;

                public function __construct($subject, $body, $isHtml, $fromEmail, $fromName, $attachments)
                {
                    $this->subject = $subject;
                    $this->body = $body;
                    $this->isHtml = $isHtml;
                    $this->fromEmail = $fromEmail;
                    $this->fromName = $fromName;
                    $this->attachments = $attachments;
                }


                public function build()
                {
                    $mail = $this->subject($this->subject);
                    
                    if (!empty($this->fromEmail)) {
                        $mail->from($this->fromEmail, $this->fromName);
                    }
                    
                    if ($this->isHtml) {
                        $mail->html($this->body);
                    } else {
                        // For plain text, use html() with stripped tags
                        $mail->html(nl2br(strip_tags($this->body)));
                    }
                    
                    return $mail;
                }

                public function attachments()
                {
                    $attachments = [];
                    foreach ($this->attachments as $attachment) {
                        if (isset($attachment['path']) && file_exists($attachment['path'])) {
                            $attachments[] = Attachment::fromPath($attachment['path'])
                                ->as($attachment['name'] ?? basename($attachment['path']));
                        }
                    }
                    return $attachments;
                }
            };

            // Configure SMTP based on frontend settings
            if (!empty($smtpConfig) && $smtpConfig['useCustom'] === true) {
                // Use custom SMTP from frontend configuration
                config([
                    'mail.mailers.smtp' => [
                        'transport' => 'smtp',
                        'host' => $smtpConfig['host'] ?? 'smtp.gmail.com',
                        'port' => $smtpConfig['port'] ?? 587,
                        'encryption' => $smtpConfig['encryption'] ?? 'tls',
                        'username' => $smtpConfig['username'] ?? '',
                        'password' => $smtpConfig['password'] ?? '',
                        'timeout' => $smtpConfig['timeout'] ?? 30,
                        'verify_peer' => false,
                        'verify_peer_name' => false,
                        'allow_self_signed' => true,
                    ]
                ]);
                
                Log::debug('📧 Custom SMTP configuration applied', [
                    'host' => $smtpConfig['host'] ?? 'smtp.gmail.com',
                    'port' => $smtpConfig['port'] ?? 587,
                    'encryption' => $smtpConfig['encryption'] ?? 'tls',
                    'username' => $smtpConfig['username'] ?? '',
                    'useCustom' => $smtpConfig['useCustom']
                ]);
            } else {
                // Use default mailer configuration from .env
                Log::debug('📧 Using default mailer configuration from .env', [
                    'default_mailer' => config('mail.default'),
                    'smtp_config' => config('mail.mailers.smtp'),
                    'useCustom' => $smtpConfig['useCustom'] ?? false
                ]);
            }
            
            $mailer = Mail::mailer();
            
            // Ensure $to is defined and not empty
            if (empty($to)) {
                Log::error('❌ Recipient email is empty before sending', [
                    'to' => $to,
                    'original_to' => $originalTo ?? 'not_set'
                ]);
                return ['success' => false, 'error' => 'Recipient email is required'];
            }
            
            Log::debug('📧 Sending email', [
                'to' => $to,
                'subject' => $subject,
                'fromEmail' => $fromEmail
            ]);
            
            if (!empty($cc)) {
                $ccEmails = array_map('trim', explode(',', $cc));
                $mailable->cc($ccEmails);
            }
            
            if (!empty($bcc)) {
                $bccEmails = array_map('trim', explode(',', $bcc));
                $mailable->bcc($bccEmails);
            }

            // Final safety check before sending
            if (!isset($to) || empty($to)) {
                Log::error('❌ Recipient email is undefined or empty', [
                    'to_isset' => isset($to),
                    'to_value' => $to ?? 'undefined',
                    'original_to' => $originalTo ?? 'not_set'
                ]);
                return ['success' => false, 'error' => 'Recipient email is undefined or empty'];
            }

            $result = $mailer->to($to)->send($mailable);

            // Store response in contact state
            if ($responseVar !== '') {
                $response = [
                    'success' => true,
                    'sent_at' => now()->toISOString(),
                    'to' => $to,
                    'subject' => $subject,
                    'cc' => $cc,
                    'bcc' => $bcc
                ];
                $contact->setContactState($this->flow_id, $responseVar, json_encode($response));
                Log::info('✅ Stored email response in contact state', ['variable' => $responseVar]);
            }

            // Continue to next node
            $nextNode = $this->getNextNodeId();
            if ($nextNode) {
                $nextNode->process($message, $data);
            }

            return ['success' => true, 'result' => $result];

        } catch (\Exception $e) {
            Log::error('❌ SendEmail node error', [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }


    protected function getNextNodeId($data = null)
    {
        if (!empty($this->outgoingEdges)) {
            return $this->outgoingEdges[0]->getTarget();
        }
        return null;
    }
}