<?php
namespace Modules\Flowmaker\Models\Nodes;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Modules\Flowmaker\Models\Contact;

class NewRequest extends Node
{
    public function process($message, $data)
    {
        Log::info('🔄 Processing NewRequest node', [
            'node_id' => $this->id ?? 'unknown',
            'flow_id' => $this->flow_id ?? 'unknown'
        ]);

        try {
            // Load settings
            $nodeData = $this->getDataAsArray();
            $settings = $nodeData['settings']['new_request'] ?? [];
            Log::debug('📋 Node settings', ['settings' => $settings]);

            // Resolve contact
            $contactId = is_object($data) ? ($data->contact_id ?? null) : ($data['contact_id'] ?? null);
            $contact = $contactId ? Contact::find($contactId) : null;
            if (!$contact) {
                Log::error('❌ Contact not found', ['contactId' => $contactId]);
                return ['success' => false];
            }
            Log::debug('👤 Contact', ['id' => $contact->id]);

            // Extract normalized settings
            $method       = strtoupper($settings['method'] ?? 'GET');
            $urlRaw       = $settings['url'] ?? '';
            $headersJson  = $settings['headers'] ?? '{}';
            $bodyRaw      = $settings['body'] ?? '';
            $bodyType     = $settings['bodyType'] ?? 'json';
            $responseVar  = trim($settings['responseVar'] ?? '');
            $timeout      = (int)($settings['timeout'] ?? 10);
            $retries      = (int)($settings['retry'] ?? 0);
            $queryParams  = $settings['queryParams'] ?? [];
            $enableQP     = (bool)($settings['enableQueryParams'] ?? false);

            // Interpolate URL
            $url = $contact->changeVariables($urlRaw, $this->flow_id);
            if (empty($url)) {
                Log::error('❌ URL is empty after interpolation', ['original' => $urlRaw]);
                return ['success' => false];
            }

            // Apply query params to URL when enabled
            if ($enableQP && !empty($queryParams) && is_array($queryParams)) {
                $url = $this->addQueryParamsToUrl($url, $queryParams, $contact);
            }

            // Parse and interpolate headers
            $headers = $this->processHeaders($headersJson, $contact);

            // Add Basic Auth if set on settings
            $basicAuthHeader = $this->handleBasicAuth($settings, $contact);
            if ($basicAuthHeader) {
                $headers['Authorization'] = $basicAuthHeader;
                Log::info('🔐 Basic Auth header added');
            }

            // Prepare request body for non-GET methods
            $body = $this->prepareRequestBody($bodyRaw, $bodyType, $contact);

            // Create HTTP client
            $http = Http::timeout($timeout)->withHeaders($headers);
            if ($retries > 0) {
                $http = $http->retry($retries, 1000);
            }

            Log::info('🌐 Making HTTP request', [
                'method'   => $method,
                'url'      => $url,
                'bodyType' => $bodyType,
                'timeout'  => $timeout,
                'retries'  => $retries
            ]);

            // Execute
            $response = $this->dispatchRequest($http, $method, $url, $body, $bodyType);

            if (!$response) {
                Log::error('❌ No response received from HTTP client');
                return ['success' => false];
            }

            $status = $response->status();
            Log::info(($status >= 200 && $status < 300) ? '✅ Request successful' : '⚠️ Non-success status', [
                'status' => $status,
                'url'    => $url
            ]);

            // Store response states - UPDATED: Store both dotted and underscore notation
            if ($responseVar !== '') {
                // Get response data
                $responseBody = $response->body();
                $responseHeaders = $response->headers();
                $responseJson = null;

                // Attempt JSON parse
                try {
                    $responseJson = $response->json();
                } catch (\Throwable $e) {
                    $responseJson = null;
                }

                // Store main composite response as JSON string
                $responsePayload = [
                    'status'  => $status,
                    'data'    => $responseBody,  // Changed from 'body' to 'data'
                    'json'    => $responseJson,
                    'headers' => $responseHeaders
                ];

                // Store main response object (used by builders)
                $this->safeSetContactState($contact, $this->flow_id, $responseVar, $responsePayload);

                // Store DOTTED notation variables (used by flow builders)
                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '.status', $status);
                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '.data', $responseBody);
                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '.headers', $responseHeaders);
                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '.json', $responseJson);

                // Store UNDERSCORE notation variables (used by execution engine)
                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_status', $status);
                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_data', $responseBody);  // Changed from '_body' to '_data'
                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_body', $responseBody);   // Keep _body for backward compatibility
                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_headers', $responseHeaders);
                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_json', $responseJson);

                // Process JSON response for additional variables
                if (!is_null($responseJson)) {
                    // Derive totals typical to HubSpot search responses
                    if (is_array($responseJson)) {
                        if (array_key_exists('total', $responseJson) && is_numeric($responseJson['total'])) {
                            $total = (int)$responseJson['total'];
                            // Store in both formats
                            $this->safeSetContactState($contact, $this->flow_id, $responseVar . '.total', $total);
                            $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_total', $total);
                            
                            $hasResults = $total > 0 ? 1 : 0;
                            $this->safeSetContactState($contact, $this->flow_id, $responseVar . '.has_results', $hasResults);
                            $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_has_results', $hasResults);
                        }
                        if (array_key_exists('results', $responseJson) && is_array($responseJson['results'])) {
                            $count = count($responseJson['results']);
                            // Store in both formats
                            $this->safeSetContactState($contact, $this->flow_id, $responseVar . '.results_count', $count);
                            $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_results_count', $count);
                            
                            if ($contact->getContactStateValue($this->flow_id, $responseVar . '_total') === null) {
                                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '.total', $count);
                                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_total', $count);
                            }
                            
                            $hasResults = $count > 0 ? 1 : 0;
                            $this->safeSetContactState($contact, $this->flow_id, $responseVar . '.has_results', $hasResults);
                            $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_has_results', $hasResults);
                        }
                    }

                    // WooCommerce product convenience
                    if ($this->isWooCommerceProductsResponse($responseJson)) {
                        $simplified = $this->extractSimplifiedProducts($responseJson);
                        $this->safeSetContactState($contact, $this->flow_id, $responseVar . '.products', $simplified);
                        $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_products', $simplified);
                    }
                }

                // Store formatted short text for WhatsApp-friendly views
                $formatted = $this->formatResponse($response, $settings);
                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '.formatted', $formatted);
                $this->safeSetContactState($contact, $this->flow_id, $responseVar . '_formatted', $formatted);

                Log::info('✅ Response variables stored', [
                    'responseVar' => $responseVar,
                    'status' => $status,
                    'hasJson' => !is_null($responseJson),
                    'bodyLength' => strlen($responseBody)
                ]);
            }

            // Continue to next node (non-blocking error-safe)
            try {
                $nextNode = $this->getNextNodeId();
                if ($nextNode) {
                    Log::info('🔄 Continue to next node', [
                        'current_node' => $this->id,
                        'next_node'    => $nextNode->id ?? 'unknown'
                    ]);
                    $nextNode->process($message, $data);
                } else {
                    Log::warning('⚠️ No next node attached', ['current_node' => $this->id]);
                }
            } catch (\Throwable $e) {
                Log::error('❌ Error continuing to next node', [
                    'current_node' => $this->id,
                    'error'        => $e->getMessage()
                ]);
            }

            return ['success' => true];

        } catch (\Throwable $e) {
            Log::error('❌ Error in NewRequest::process', [
                'error' => $e->getMessage(),
            ]);
            return ['success' => false];
        }
    }

    /**
     * Safely store contact state, converting arrays to JSON strings
     */
    protected function safeSetContactState($contact, $flowId, $key, $value)
    {
        try {
            // Convert arrays/objects to JSON strings to prevent "Array to string conversion"
            if (is_array($value) || is_object($value)) {
                $value = json_encode($value);
            }
            
            $contact->setContactState($flowId, $key, $value);
        } catch (\Throwable $e) {
            Log::error('❌ Error storing contact state', [
                'key' => $key,
                'error' => $e->getMessage()
            ]);
        }
    }

    /* =========================
       Request assembly helpers
       ========================= */

    protected function processHeaders(string $headersJson, Contact $contact): array
    {
        $headers = [];
        try {
            $parsed = json_decode($headersJson, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                Log::error('❌ Invalid headers JSON', ['error' => json_last_error_msg()]);
                return $headers;
            }
            foreach (($parsed ?? []) as $k => $v) {
                $headers[$k] = $this->processHeaderValue((string)$v, $contact);
            }
        } catch (\Throwable $e) {
            Log::error('❌ Error processing headers', ['error' => $e->getMessage()]);
        }
        return $headers;
    }

    protected function processHeaderValue(string $value, Contact $contact): string
    {
        if ($this->isSensitiveHeaderValue($value) && strpos($value, '{{') === false) {
            return $value;
        }
        $interpolated = $contact->changeVariables($value, $this->flow_id);

        if (preg_match('/^Basic\s+(.+)$/', $interpolated, $m)) {
            $base = $m[1];
            $pad  = strlen($base) % 4;
            if ($pad) {
                $base .= str_repeat('=', 4 - $pad);
            }
            if (preg_match('/^[A-Za-z0-9+\/]+={0,2}$/', $base)) {
                return 'Basic ' . $base;
            }
        }

        return $interpolated;
    }

    protected function handleBasicAuth(array $settings, Contact $contact): ?string
    {
        $username = $settings['basicAuthUsername'] ?? '';
        $password = $settings['basicAuthPassword'] ?? '';
        if ($username === '' || $password === '') {
            return null;
        }
        $username = $contact->changeVariables($username, $this->flow_id);
        $password = $contact->changeVariables($password, $this->flow_id);
        $token = base64_encode($username . ':' . $password);
        $pad = strlen($token) % 4;
        if ($pad) {
            $token .= str_repeat('=', 4 - $pad);
        }
        return 'Basic ' . $token;
    }

    protected function prepareRequestBody(string $bodyRaw, string $bodyType, Contact $contact)
    {
        switch ($bodyType) {
            case 'json':
                $interpolated = $contact->changeVariables($bodyRaw, $this->flow_id);
                $data = json_decode($interpolated, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    Log::error('❌ Invalid JSON after interpolation', [
                        'error'         => json_last_error_msg(),
                        'interpolated'  => $interpolated
                    ]);
                    return [];
                }
                return $data ?? [];

            case 'form':
                $interpolated = $contact->changeVariables($bodyRaw, $this->flow_id);
                $out = [];
                foreach (preg_split("/\r\n|\n|\r/", $interpolated) as $line) {
                    $line = trim($line);
                    if ($line === '') continue;
                    $parts = explode('=', $line, 2);
                    if (count($parts) === 2) {
                        $out[trim($parts[0])] = trim($parts[1]);
                    }
                }
                return $out;

            case 'xml':
                return $contact->changeVariables($bodyRaw, $this->flow_id);

            case 'multipart':
                $interpolated = $contact->changeVariables($bodyRaw, $this->flow_id);
                $out = [];
                foreach (preg_split("/\r\n|\n|\r/", $interpolated) as $line) {
                    $line = trim($line);
                    if ($line === '') continue;
                    $parts = explode('=', $line, 2);
                    if (count($parts) === 2) {
                        $out[trim($parts[0])] = trim($parts[1]);
                    }
                }
                return $out;

            default:
                return [];
        }
    }

    protected function dispatchRequest($http, string $method, string $url, $body, string $bodyType)
    {
        switch ($method) {
            case 'GET':
                return $http->get($url);

            case 'POST':
                return $this->postLike($http, 'post', $url, $body, $bodyType);

            case 'PUT':
                return $this->postLike($http, 'put', $url, $body, $bodyType);

            case 'PATCH':
                return $this->postLike($http, 'patch', $url, $body, $bodyType);

            case 'DELETE':
                return $this->postLike($http, 'delete', $url, $body, $bodyType);

            case 'HEAD':
                return $http->head($url);

            case 'OPTIONS':
                return $http->send('OPTIONS', $url);

            default:
                Log::error('❌ Unsupported HTTP method', ['method' => $method]);
                return null;
        }
    }

    protected function postLike($http, string $verb, string $url, $body, string $bodyType)
    {
        switch ($bodyType) {
            case 'json':
                return $http->{$verb}($url, is_array($body) ? $body : []);

            case 'form':
                return $http->asForm()->{$verb}($url, is_array($body) ? $body : []);

            case 'xml':
                return $http->withHeaders(['Content-Type' => 'application/xml'])
                            ->withBody(is_string($body) ? $body : '', 'application/xml')
                            ->{$verb}($url);

            case 'multipart':
                return $http->asForm()->{$verb}($url, is_array($body) ? $body : []);

            default:
                return $http->{$verb}($url, is_array($body) ? $body : []);
        }
    }

    /* =========================
       URL helpers
       ========================= */

    protected function addQueryParamsToUrl(string $url, array $queryParams, Contact $contact): string
    {
        $parsed = parse_url($url);
        $existing = [];
        if (isset($parsed['query'])) {
            parse_str($parsed['query'], $existing);
        }

        foreach ($queryParams as $param) {
            if (!empty($param['key'])) {
                $k = $contact->changeVariables($param['key'], $this->flow_id);
                $v = isset($param['value']) ? $contact->changeVariables($param['value'], $this->flow_id) : '';
                $existing[$k] = $v;
            }
        }

        $parsed['query'] = http_build_query($existing);
        return $this->buildUrl($parsed);
    }

    protected function buildUrl(array $parts): string
    {
        $url = '';
        if (!empty($parts['scheme'])) $url .= $parts['scheme'] . '://';
        if (!empty($parts['host']))   $url .= $parts['host'];
        if (!empty($parts['port']))   $url .= ':' . $parts['port'];
        if (!empty($parts['path']))   $url .= $parts['path'];
        if (!empty($parts['query']))  $url .= '?' . $parts['query'];
        if (!empty($parts['fragment'])) $url .= '#' . $parts['fragment'];
        return $url;
    }

    /* =========================
       Formatting helpers
       ========================= */

    protected function formatResponse($response, array $settings): string
    {
        $formatType     = $settings['formatType'] ?? 'json';
        $formatTemplate = $settings['formatTemplate'] ?? '';

        if ($formatType === 'custom' && $formatTemplate !== '') {
            $json = null;
            try { $json = $response->json(); } catch (\Throwable $e) {}
            if (is_array($json)) {
                $formatted = $formatTemplate;
                foreach ($json as $k => $v) {
                    if (is_scalar($v)) {
                        $formatted = str_replace('{{' . $k . '}}', (string)$v, $formatted);
                    }
                }
                return $this->limitResponseForWhatsApp($formatted);
            }
        }

        $json = null;
        try { $json = $response->json(); } catch (\Throwable $e) {}

        if (is_array($json)) {
            if ($this->isWooCommerceProductsResponse($json)) {
                return $this->formatWooCommerceProducts($json);
            }

            if (isset($json['results']) && is_array($json['results'])) {
                $lines = ["🧾 Results:"];
                $i = 1;
                foreach ($json['results'] as $row) {
                    $label = 'Item ' . $i;
                    if (is_array($row)) {
                        if (isset($row['id']))   { $label .= ' (id: ' . $row['id'] . ')'; }
                        if (isset($row['name'])) { $label .= ' ' . $row['name']; }
                    }
                    $lines[] = $label;
                    $i++;
                    if (count($lines) > 20) {
                        $lines[] = '...and more.';
                        break;
                    }
                }
                return $this->limitResponseForWhatsApp(implode("\n", $lines));
            }

            $lines = ["🛒 Response Data:"];
            $n = 1;
            foreach ($json as $item) {
                if (is_array($item)) {
                    $lines[] = $n . '. ' . ($item['name'] ?? $item['title'] ?? 'Unknown');
                } else {
                    $lines[] = $n . '. ' . (string)$item;
                }
                $n++;
                if ($n > 15) {
                    $lines[] = '...and more.';
                    break;
                }
            }
            return $this->limitResponseForWhatsApp(implode("\n", $lines));
        }

        return $this->limitResponseForWhatsApp('Response: ' . $response->body());
    }

    protected function isWooCommerceProductsResponse($json): bool
    {
        if (!is_array($json) || !isset($json[0]) || !is_array($json)) {
            return false;
        }
        $first = $json;
        return isset($first['id'], $first['name'], $first['type']);
    }

    protected function formatWooCommerceProducts(array $products): string
    {
        if (empty($products)) {
            return "❌ No products found.";
        }
        $formatted = "🛒 Products Found:\n\n";
        $maxItems  = $this->calculateMaxProductsForWhatsApp();
        $shown     = 0;
        $maxLen    = 3800;

        foreach ($products as $idx => $p) {
            if ($shown >= $maxItems) {
                $formatted .= "\n📋 ...and " . (count($products) - $shown) . " more products.\n";
                break;
            }
            $formatted .= '**' . ($idx + 1) . '. ' . ($p['name'] ?? 'Unnamed') . "**\n";
            if (isset($p['id']))    $formatted .= "🆔 " . $p['id'] . "\n";
            if (isset($p['price'])) $formatted .= "💰 $" . $p['price'] . "\n";
            if (isset($p['stock_status'])) {
                $formatted .= ($p['stock_status'] === 'instock' ? '✅ In Stock' : '❌ Out of Stock') . "\n";
            }
            $formatted .= "\n";
            $shown++;

            if (strlen($formatted) > $maxLen) {
                $formatted .= "\n📋 Response truncated.\n";
                break;
            }
        }
        return $this->limitResponseForWhatsApp($formatted);
    }

    protected function extractSimplifiedProducts(array $products): array
    {
        $out = [];
        foreach ($products as $p) {
            $out[] = [
                'id'                => $p['id'] ?? null,
                'name'              => $p['name'] ?? '',
                'price'             => $p['price'] ?? '',
                'description'       => isset($p['description']) ? strip_tags($p['description']) : '',
                'short_description' => isset($p['short_description']) ? strip_tags($p['short_description']) : '',
                'stock_status'      => $p['stock_status'] ?? 'unknown',
                'permalink'         => $p['permalink'] ?? '',
                'sku'               => $p['sku'] ?? '',
                'type'              => $p['type'] ?? 'simple'
            ];
        }
        return $out;
    }

    protected function calculateMaxProductsForWhatsApp(): int
    {
        $charsPer = 140;
        $buffer   = 800;
        $avail    = 3800 - $buffer;
        $max      = (int)floor($avail / $charsPer);
        return max(3, min($max, 8));
    }

    protected function limitResponseForWhatsApp(string $text, int $maxLength = 4096): string
    {
        if (strlen($text) <= $maxLength) {
            return $text;
        }
        $trunc = substr($text, 0, $maxLength - 100);
        $last  = strrpos($trunc, "\n");
        if ($last !== false) {
            $trunc = substr($trunc, 0, $last);
        }
        $trunc .= "\n\n📋 Response truncated due to length.\n";
        return $trunc;
    }

    /* =========================
       Security helpers
       ========================= */

    protected function isSensitiveHeaderValue(string $value): bool
    {
        if (preg_match('/^Basic\s+[A-Za-z0-9+\/]+={0,2}$/', $value)) return true;
        if (preg_match('/^Bearer\s+[A-Za-z0-9\-._~+\/=]+$/', $value)) return true;
        if (preg_match('/^(ck_|cs_|pk_|sk_|rk_)[A-Za-z0-9_]+$/', $value)) return true;
        if (strlen($value) > 50 && preg_match('/^[A-Za-z0-9+\/=_-]+$/', $value)) return true;
        return false;
    }

    /* =========================
       Flow helpers
       ========================= */

    protected function getNextNodeId($data = null)
    {
        Log::debug('🔍 Getting next node', [
            'current_node' => $this->id,
            'outgoing_edges_count' => count($this->outgoingEdges ?? [])
        ]);

        if (!empty($this->outgoingEdges)) {
            $next = $this->outgoingEdges[0]->getTarget();
            Log::debug('✅ Next node found', ['next_node' => $next ? $next->id : 'null']);
            return $next;
        }

        Log::debug('🔄 Falling back to parent getNextNodeId');
        return parent::getNextNodeId($data);
    }
}
