import React, { useState } from 'react';
import { Handle, Position, NodeProps } from '@xyflow/react';
import { Network, Settings, Play } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

interface VoiceFlowNodeProps {
  data: any;
  id: string;
  selected?: boolean;
  isConnectable?: boolean;
}

const VoiceFlowNode: React.FC<VoiceFlowNodeProps> = ({ data, selected }) => {
  const [isEditing, setIsEditing] = useState(false);
  const [settings, setSettings] = useState(data.settings || {
    apiKey: '',
    assistantId: '',
    versionId: '',
    sessionId: '',
    message: '',
    variables: [],
    timeout: 30,
    retryCount: 3,
    responseVar: 'voiceflow_response'
  });

  const handleSave = () => {
    data.settings = settings;
    setIsEditing(false);
  };

  const addVariable = () => {
    setSettings(prev => ({
      ...prev,
      variables: [...prev.variables, { key: '', value: '' }]
    }));
  };

  const updateVariable = (index: number, field: 'key' | 'value', value: string) => {
    setSettings(prev => ({
      ...prev,
      variables: prev.variables.map((v, i) => 
        i === index ? { ...v, [field]: value } : v
      )
    }));
  };

  const removeVariable = (index: number) => {
    setSettings(prev => ({
      ...prev,
      variables: prev.variables.filter((_, i) => i !== index)
    }));
  };

  return (
    <div className={`px-4 py-2 shadow-md rounded-md bg-violet-100 border-2 min-w-[300px] ${
      selected ? 'border-violet-500' : 'border-violet-200'
    }`}>
      <Handle type="target" position={Position.Top} className="w-3 h-3" />
      
      <div className="flex items-center gap-2 mb-2">
        <Network className="w-4 h-4 text-violet-600" />
        <div className="font-bold text-violet-800">VoiceFlow AI</div>
        <Button
          variant="ghost"
          size="sm"
          onClick={() => setIsEditing(!isEditing)}
          className="ml-auto p-1 h-6 w-6"
        >
          <Settings className="w-3 h-3" />
        </Button>
      </div>

      {isEditing ? (
        <div className="space-y-3">
          <div>
            <Label htmlFor="apiKey">API Key</Label>
            <Input
              id="apiKey"
              value={settings.apiKey}
              onChange={(e) => setSettings(prev => ({ ...prev, apiKey: e.target.value }))}
              placeholder="Enter VoiceFlow API key"
              type="password"
            />
          </div>

          <div>
            <Label htmlFor="assistantId">Assistant ID</Label>
            <Input
              id="assistantId"
              value={settings.assistantId}
              onChange={(e) => setSettings(prev => ({ ...prev, assistantId: e.target.value }))}
              placeholder="Enter Assistant ID"
            />
          </div>

          <div>
            <Label htmlFor="versionId">Version ID</Label>
            <Input
              id="versionId"
              value={settings.versionId}
              onChange={(e) => setSettings(prev => ({ ...prev, versionId: e.target.value }))}
              placeholder="Enter Version ID"
            />
          </div>

          <div>
            <Label htmlFor="sessionId">Session ID</Label>
            <Input
              id="sessionId"
              value={settings.sessionId}
              onChange={(e) => setSettings(prev => ({ ...prev, sessionId: e.target.value }))}
              placeholder="Enter Session ID"
            />
          </div>

          <div>
            <Label htmlFor="message">Message</Label>
            <Textarea
              id="message"
              value={settings.message}
              onChange={(e) => setSettings(prev => ({ ...prev, message: e.target.value }))}
              placeholder="Enter message to send to VoiceFlow"
              rows={3}
            />
          </div>

          <div>
            <Label>Variables</Label>
            {settings.variables.map((variable, index) => (
              <div key={index} className="flex gap-2 mb-2">
                <Input
                  value={variable.key}
                  onChange={(e) => updateVariable(index, 'key', e.target.value)}
                  placeholder="Key"
                />
                <Input
                  value={variable.value}
                  onChange={(e) => updateVariable(index, 'value', e.target.value)}
                  placeholder="Value"
                />
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => removeVariable(index)}
                  className="px-2"
                >
                  ×
                </Button>
              </div>
            ))}
            <Button variant="outline" size="sm" onClick={addVariable} className="w-full">
              Add Variable
            </Button>
          </div>

          <div className="flex gap-2">
            <div className="flex-1">
              <Label htmlFor="timeout">Timeout (seconds)</Label>
              <Input
                id="timeout"
                type="number"
                value={settings.timeout}
                onChange={(e) => setSettings(prev => ({ ...prev, timeout: parseInt(e.target.value) || 30 }))}
              />
            </div>
            <div className="flex-1">
              <Label htmlFor="retryCount">Retry Count</Label>
              <Input
                id="retryCount"
                type="number"
                value={settings.retryCount}
                onChange={(e) => setSettings(prev => ({ ...prev, retryCount: parseInt(e.target.value) || 3 }))}
              />
            </div>
          </div>

          <div>
            <Label htmlFor="responseVar">Response Variable</Label>
            <Input
              id="responseVar"
              value={settings.responseVar}
              onChange={(e) => setSettings(prev => ({ ...prev, responseVar: e.target.value }))}
              placeholder="Variable name to store VoiceFlow response"
            />
          </div>

          <div className="flex gap-2">
            <Button onClick={handleSave} size="sm" className="flex-1">
              Save
            </Button>
            <Button 
              variant="outline" 
              onClick={() => setIsEditing(false)} 
              size="sm" 
              className="flex-1"
            >
              Cancel
            </Button>
          </div>
        </div>
      ) : (
        <div className="text-sm text-violet-700">
          <div className="mb-1">
            <strong>Assistant:</strong> {settings.assistantId || 'Not configured'}
          </div>
          <div className="mb-1">
            <strong>Version:</strong> {settings.versionId || 'Not configured'}
          </div>
          <div className="mb-1">
            <strong>Session:</strong> {settings.sessionId || 'Auto-generated'}
          </div>
          {settings.message && (
            <div className="mb-1">
              <strong>Message:</strong> {settings.message.substring(0, 50)}
              {settings.message.length > 50 ? '...' : ''}
            </div>
          )}
          {settings.variables.length > 0 && (
            <div>
              <strong>Variables:</strong> {settings.variables.length} configured
            </div>
          )}
        </div>
      )}

      <Handle type="source" position={Position.Bottom} className="w-3 h-3" />
    </div>
  );
};

export default VoiceFlowNode;
