import React, { useState, useEffect, useCallback } from 'react';
import { Handle, Position } from '@xyflow/react';
import { 
  FileSpreadsheet, 
  Settings, 
  Info,
  ExternalLink,
  Key,
  Globe,
  Trash2
} from 'lucide-react';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Button } from '@/components/ui/button';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { VariableInput } from '../common/VariableInput';
import { VariableTextArea } from '../common/VariableTextArea';
import { useFlowActions } from '@/hooks/useFlowActions';
import {
  ContextMenu,
  ContextMenuContent,
  ContextMenuItem,
  ContextMenuTrigger,
} from "@/components/ui/context-menu";

interface SheetsNodeProps {
  data: any;
  id: string;
  selected?: boolean;
  isConnectable?: boolean;
}

interface SheetsSettings {
  operation: 'read' | 'write' | 'update' | 'create_sheet';
  spreadsheetId: string;
  range: string;
  authMethod: 'service_account' | 'api_key';
  serviceAccountPath: string;
  apiKey: string;
  responseVar: string;
  dataToWrite: string;
  writeMode: 'append' | 'overwrite';
  sheetName: string;
}

const SheetsNode = ({ data, id, selected, isConnectable }: SheetsNodeProps) => {
  const { deleteNode } = useFlowActions();
  const [nodeSettings, setNodeSettings] = useState<SheetsSettings>(() => {
    const initialSettings: SheetsSettings = {
      operation: data.settings?.operation || 'read',
      spreadsheetId: data.settings?.spreadsheetId || '',
      range: data.settings?.range || 'A:Z',
      authMethod: data.settings?.authMethod || 'service_account',
      serviceAccountPath: data.settings?.serviceAccountPath || 'storage/google-credentials.json',
      apiKey: data.settings?.apiKey || '',
      responseVar: data.settings?.responseVar || 'sheets_response',
      dataToWrite: data.settings?.dataToWrite || '',
      writeMode: data.settings?.writeMode || 'append',
      sheetName: data.settings?.sheetName || 'New Sheet'
    };
    return initialSettings;
  });

  useEffect(() => {
    if (data.settings) {
      Object.keys(nodeSettings).forEach(key => {
        data.settings[key] = nodeSettings[key];
      });
    }
  }, [nodeSettings, data]);

  const updateNodeData = useCallback((key: string, value: any) => {
    setNodeSettings(prev => ({
      ...prev,
      [key]: value
    }));
  }, []);

  const getSpreadsheetIdFromUrl = (url: string) => {
    const match = url.match(/\/spreadsheets\/d\/([a-zA-Z0-9-_]+)/);
    return match ? match[1] : '';
  };

  const handleUrlPaste = (url: string) => {
    const spreadsheetId = getSpreadsheetIdFromUrl(url);
    if (spreadsheetId) {
      updateNodeData('spreadsheetId', spreadsheetId);
    }
  };

  return (
    <ContextMenu>
      <ContextMenuTrigger>
        <div className="bg-white rounded-lg shadow-lg w-[400px] min-h-[650px] flex flex-col">
          <Handle
            type="target"
            position={Position.Left}
            style={{ left: '-4px', background: '#555', zIndex: 50 }}
            isConnectable={isConnectable}
          />

          <div className="flex items-center gap-2 mb-4 pb-2 border-b border-gray-100 px-4 pt-3 bg-green-50">
            <FileSpreadsheet className="h-4 w-4 text-green-600" />
            <div className="font-medium">Google Sheets</div>
          </div>

          <div className="p-4 space-y-4 flex-1 overflow-y-auto">
        <Tabs defaultValue="configuration" className="w-full">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="configuration">Configuration</TabsTrigger>
            <TabsTrigger value="setup">Setup Guide</TabsTrigger>
          </TabsList>

          <TabsContent value="configuration" className="space-y-4">
            {/* Operation Selection */}
            <div className="space-y-2">
              <Label htmlFor="operation">Operation</Label>
              <Select
                value={nodeSettings.operation}
                onValueChange={(value) => updateNodeData('operation', value)}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="read">Read Data</SelectItem>
                  <SelectItem value="write">Write Data</SelectItem>
                  <SelectItem value="update">Update Data</SelectItem>
                  <SelectItem value="create_sheet">Create Sheet</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {/* Spreadsheet ID */}
            <div className="space-y-2">
              <Label htmlFor="spreadsheetId">Spreadsheet ID or URL</Label>
              <div className="flex space-x-2">
                <VariableInput
                  value={nodeSettings.spreadsheetId}
                  onChange={(value) => updateNodeData('spreadsheetId', value)}
                  placeholder="Enter spreadsheet ID or paste Google Sheets URL"
                  className="flex-1"
                />
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => {
                    const url = prompt('Paste Google Sheets URL:');
                    if (url) handleUrlPaste(url);
                  }}
                >
                  <ExternalLink className="w-4 h-4" />
                </Button>
              </div>
            </div>

            {/* Range (for read/write/update operations) */}
            {nodeSettings.operation !== 'create_sheet' && (
              <div className="space-y-2">
                <Label htmlFor="range">Range</Label>
                <VariableInput
                  value={nodeSettings.range}
                  onChange={(value) => updateNodeData('range', value)}
                  placeholder="A1:B10 or Sheet1!A:Z"
                />
              </div>
            )}

            {/* Sheet Name (for create_sheet operation) */}
            {nodeSettings.operation === 'create_sheet' && (
              <div className="space-y-2">
                <Label htmlFor="sheetName">Sheet Name</Label>
                <VariableInput
                  value={nodeSettings.sheetName}
                  onChange={(value) => updateNodeData('sheetName', value)}
                  placeholder="Enter new sheet name"
                />
              </div>
            )}

            {/* Authentication Method */}
            <div className="space-y-2">
              <Label htmlFor="authMethod">Authentication Method</Label>
              <Select
                value={nodeSettings.authMethod}
                onValueChange={(value) => updateNodeData('authMethod', value)}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="service_account">
                    <div className="flex items-center space-x-2">
                      <Key className="w-4 h-4" />
                      <span>Service Account</span>
                    </div>
                  </SelectItem>
                  <SelectItem value="api_key">
                    <div className="flex items-center space-x-2">
                      <Globe className="w-4 h-4" />
                      <span>API Key (Read Only)</span>
                    </div>
                  </SelectItem>
                </SelectContent>
              </Select>
            </div>

            {/* Service Account Path */}
            {nodeSettings.authMethod === 'service_account' && (
              <div className="space-y-2">
                <Label htmlFor="serviceAccountPath">Service Account File Path</Label>
                <Input
                  value={nodeSettings.serviceAccountPath}
                  onChange={(e) => updateNodeData('serviceAccountPath', e.target.value)}
                  placeholder="storage/google-credentials.json"
                />
              </div>
            )}

            {/* API Key */}
            {nodeSettings.authMethod === 'api_key' && (
              <div className="space-y-2">
                <Label htmlFor="apiKey">API Key</Label>
                <Input
                  type="password"
                  value={nodeSettings.apiKey}
                  onChange={(e) => updateNodeData('apiKey', e.target.value)}
                  placeholder="Enter your Google API key"
                />
              </div>
            )}

            {/* Data to Write (for write/update operations) */}
            {(nodeSettings.operation === 'write' || nodeSettings.operation === 'update') && (
              <div className="space-y-2">
                <Label htmlFor="dataToWrite">Data to Write</Label>
                <VariableTextArea
                  value={nodeSettings.dataToWrite}
                  onChange={(value) => updateNodeData('dataToWrite', value)}
                  placeholder="Enter JSON array, CSV data, or simple text"
                  rows={4}
                />
                <div className="text-sm text-gray-500">
                  <p>Supported formats:</p>
                  <ul className="list-disc list-inside ml-2">
                    <li>JSON: [["Name", "Email"], ["John", "john@example.com"]]</li>
                    <li>CSV: Name,Email\nJohn,john@example.com</li>
                    <li>Simple text: Single value</li>
                  </ul>
                </div>
              </div>
            )}

            {/* Write Mode (for write operation) */}
            {nodeSettings.operation === 'write' && (
              <div className="space-y-2">
                <Label htmlFor="writeMode">Write Mode</Label>
                <Select
                  value={nodeSettings.writeMode}
                  onValueChange={(value) => updateNodeData('writeMode', value)}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="append">Append to Sheet</SelectItem>
                    <SelectItem value="overwrite">Overwrite Range</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            )}

            {/* Response Variable */}
            <div className="space-y-2">
              <Label htmlFor="responseVar">Response Variable</Label>
              <VariableInput
                value={nodeSettings.responseVar}
                onChange={(value) => updateNodeData('responseVar', value)}
                placeholder="Variable name to store response"
              />
            </div>
          </TabsContent>

          <TabsContent value="setup" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Settings className="w-5 h-5" />
                  <span>Setup Guide</span>
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <Alert>
                  <Info className="h-4 w-4" />
                  <AlertDescription>
                    Follow these steps to set up Google Sheets integration
                  </AlertDescription>
                </Alert>

                <div className="space-y-4">
                  <div>
                    <h4 className="font-semibold mb-2">Step 1: Create Google Cloud Project</h4>
                    <ol className="list-decimal list-inside space-y-1 text-sm">
                      <li>Go to <a href="https://console.cloud.google.com" target="_blank" className="text-blue-600">Google Cloud Console</a></li>
                      <li>Create a new project or select existing one</li>
                      <li>Enable Google Sheets API</li>
                    </ol>
                  </div>

                  <div>
                    <h4 className="font-semibold mb-2">Step 2: Set up Authentication</h4>
                    <div className="space-y-2">
                      <p className="text-sm font-medium">For Service Account (Recommended):</p>
                      <ol className="list-decimal list-inside space-y-1 text-sm ml-4">
                        <li>Go to "Credentials" in Google Cloud Console</li>
                        <li>Create "Service Account"</li>
                        <li>Download JSON key file</li>
                        <li>Upload to your server: <code className="bg-gray-100 px-1 rounded">storage/google-credentials.json</code></li>
                        <li>Share your Google Sheet with the service account email</li>
                      </ol>
                    </div>
                    <div className="space-y-2 mt-3">
                      <p className="text-sm font-medium">For API Key (Read Only):</p>
                      <ol className="list-decimal list-inside space-y-1 text-sm ml-4">
                        <li>Go to "Credentials" in Google Cloud Console</li>
                        <li>Create "API Key"</li>
                        <li>Restrict to Google Sheets API</li>
                        <li>Copy the API key</li>
                      </ol>
                    </div>
                  </div>

                  <div>
                    <h4 className="font-semibold mb-2">Step 3: Get Spreadsheet ID</h4>
                    <ol className="list-decimal list-inside space-y-1 text-sm">
                      <li>Open your Google Sheet</li>
                      <li>Copy the URL or extract ID from URL</li>
                      <li>URL format: <code className="bg-gray-100 px-1 rounded">https://docs.google.com/spreadsheets/d/SPREADSHEET_ID/edit</code></li>
                    </ol>
                  </div>

                  <div>
                    <h4 className="font-semibold mb-2">Step 4: Configure Node</h4>
                    <ol className="list-decimal list-inside space-y-1 text-sm">
                      <li>Select operation (Read/Write/Update/Create)</li>
                      <li>Enter spreadsheet ID or paste URL</li>
                      <li>Set range (e.g., A1:B10, Sheet1!A:Z)</li>
                      <li>Configure authentication method</li>
                      <li>Set response variable name</li>
                    </ol>
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
          </div>

          <Handle
            type="source"
            position={Position.Right}
            style={{ right: '-4px', background: '#555', zIndex: 50 }}
            isConnectable={isConnectable}
          />
        </div>
      </ContextMenuTrigger>
      <ContextMenuContent>
        <ContextMenuItem
          className="text-red-600 focus:text-red-600 focus:bg-red-100"
          onClick={() => deleteNode(id)}
        >
          <Trash2 className="mr-2 h-4 w-4" />
          Delete
        </ContextMenuItem>
      </ContextMenuContent>
    </ContextMenu>
  );
};

export default SheetsNode;