import React, { useState, useEffect, useCallback } from 'react';
import { Handle, Position } from '@xyflow/react';
import { 
  Mail, 
  Settings, 
  Paperclip, 
  Plus,
  Trash2,
  Info,
  User,
  Users,
  FileText
} from 'lucide-react';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Button } from '@/components/ui/button';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Switch } from '@/components/ui/switch';
import { VariableInput } from '../common/VariableInput';
import { VariableTextArea } from '../common/VariableTextArea';
import { useFlowActions } from '@/hooks/useFlowActions';
import {
  ContextMenu,
  ContextMenuContent,
  ContextMenuItem,
  ContextMenuTrigger,
} from "@/components/ui/context-menu";

interface SendEmailNodeProps {
  data: any;
  id: string;
  selected?: boolean;
  isConnectable?: boolean;
}

interface Attachment {
  id: string;
  name: string;
  path: string;
}

interface SMTPConfig {
  useCustom: boolean;
  host: string;
  port: number;
  encryption: 'tls' | 'ssl';
  username: string;
  password: string;
  timeout: number;
}

interface SendEmailSettings {
  to: string;
  cc: string;
  bcc: string;
  subject: string;
  body: string;
  isHtml: boolean;
  fromEmail: string;
  fromName: string;
  responseVar: string;
  attachments: Attachment[];
  smtpConfig: SMTPConfig;
}

const SendEmailNode = ({ data, id, selected, isConnectable }: SendEmailNodeProps) => {
  const { deleteNode } = useFlowActions();
  const [nodeSettings, setNodeSettings] = useState<SendEmailSettings>(() => {
    const initialSettings: SendEmailSettings = {
      to: data.settings?.to || '',
      cc: data.settings?.cc || '',
      bcc: data.settings?.bcc || '',
      subject: data.settings?.subject || '',
      body: data.settings?.body || '',
      isHtml: data.settings?.isHtml ?? true,
      fromEmail: data.settings?.fromEmail || '',
      fromName: data.settings?.fromName || '',
      responseVar: data.settings?.responseVar || 'email_response',
      attachments: data.settings?.attachments || [],
      smtpConfig: data.settings?.smtpConfig || {
        useCustom: false,
        host: 'smtp.gmail.com',
        port: 587,
        encryption: 'tls',
        username: '',
        password: '',
        timeout: 30
      }
    };
    return initialSettings;
  });

  useEffect(() => {
    if (data.settings) {
      Object.keys(nodeSettings).forEach(key => {
        data.settings[key] = nodeSettings[key];
      });
    }
  }, [nodeSettings, data]);

  const updateNodeData = useCallback((key: string, value: any) => {
    setNodeSettings(prev => ({
      ...prev,
      [key]: value
    }));
  }, []);

  const addAttachment = () => {
    const newAttachment: Attachment = {
      id: Date.now().toString(),
      name: '',
      path: ''
    };
    updateNodeData('attachments', [...nodeSettings.attachments, newAttachment]);
  };

  const removeAttachment = (id: string) => {
    updateNodeData('attachments', nodeSettings.attachments.filter(att => att.id !== id));
  };

  const updateAttachment = (id: string, field: keyof Attachment, value: string) => {
    updateNodeData('attachments', nodeSettings.attachments.map(att => 
      att.id === id ? { ...att, [field]: value } : att
    ));
  };

  return (
    <ContextMenu>
      <ContextMenuTrigger>
        <div className="bg-white rounded-lg shadow-lg w-[400px] min-h-[650px] flex flex-col">
          <Handle
            type="target"
            position={Position.Left}
            style={{ left: '-4px', background: '#555', zIndex: 50 }}
            isConnectable={isConnectable}
          />

          <div className="flex items-center gap-2 mb-4 pb-2 border-b border-gray-100 px-4 pt-3 bg-blue-50">
            <Mail className="h-4 w-4 text-blue-600" />
            <div className="font-medium">Send Email</div>
          </div>

          <div className="p-4 space-y-4 flex-1 overflow-y-auto">
        <Tabs defaultValue="configuration" className="w-full">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="configuration">Configuration</TabsTrigger>
            <TabsTrigger value="setup">Setup Guide</TabsTrigger>
          </TabsList>

          <TabsContent value="configuration" className="space-y-4">
            {/* Recipients */}
            <div className="space-y-4">
              <h4 className="font-semibold flex items-center space-x-2">
                <Users className="w-4 h-4" />
                <span>Recipients</span>
              </h4>
              
              <div className="space-y-2">
                <Label htmlFor="to">To (Required)</Label>
                <VariableInput
                  value={nodeSettings.to}
                  onChange={(value) => updateNodeData('to', value)}
                  placeholder="recipient@example.com or {{contact_email}}"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="cc">CC (Optional)</Label>
                <VariableInput
                  value={nodeSettings.cc}
                  onChange={(value) => updateNodeData('cc', value)}
                  placeholder="cc@example.com,cc2@example.com"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="bcc">BCC (Optional)</Label>
                <VariableInput
                  value={nodeSettings.bcc}
                  onChange={(value) => updateNodeData('bcc', value)}
                  placeholder="bcc@example.com,bcc2@example.com"
                />
              </div>
            </div>

            {/* Email Content */}
            <div className="space-y-4">
              <h4 className="font-semibold flex items-center space-x-2">
                <FileText className="w-4 h-4" />
                <span>Email Content</span>
              </h4>

              <div className="space-y-2">
                <Label htmlFor="subject">Subject</Label>
                <VariableInput
                  value={nodeSettings.subject}
                  onChange={(value) => updateNodeData('subject', value)}
                  placeholder="Email subject or {{dynamic_subject}}"
                />
              </div>

              <div className="space-y-2">
                <div className="flex items-center space-x-2">
                  <Label htmlFor="isHtml">HTML Email</Label>
                  <Switch
                    checked={nodeSettings.isHtml}
                    onCheckedChange={(checked) => updateNodeData('isHtml', checked)}
                  />
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="body">Email Body</Label>
                <VariableTextArea
                  value={nodeSettings.body}
                  onChange={(value) => updateNodeData('body', value)}
                  placeholder={nodeSettings.isHtml ? 
                    '<h1>Hello {{contact_name}}</h1><p>Your message here...</p>' : 
                    'Hello {{contact_name}},\n\nYour message here...'
                  }
                />
                {nodeSettings.isHtml && (
                  <div className="text-sm text-gray-500">
                    <p>HTML is supported. Use variables like {`{{contact_name}}`}, {`{{contact_email}}`}</p>
                  </div>
                )}
              </div>
            </div>

            {/* Sender Information */}
            <div className="space-y-4">
              <h4 className="font-semibold flex items-center space-x-2">
                <User className="w-4 h-4" />
                <span>Sender Information</span>
              </h4>

              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="fromEmail">From Email</Label>
                  <VariableInput
                    value={nodeSettings.fromEmail}
                    onChange={(value) => updateNodeData('fromEmail', value)}
                    placeholder="sender@example.com"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="fromName">From Name</Label>
                  <VariableInput
                    value={nodeSettings.fromName}
                    onChange={(value) => updateNodeData('fromName', value)}
                    placeholder="Your Company Name"
                  />
                </div>
              </div>
            </div>

            {/* Attachments */}
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <h4 className="font-semibold flex items-center space-x-2">
                  <Paperclip className="w-4 h-4" />
                  <span>Attachments</span>
                </h4>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={addAttachment}
                >
                  <Plus className="w-4 h-4 mr-1" />
                  Add
                </Button>
              </div>

              {nodeSettings.attachments.map((attachment) => (
                <div key={attachment.id} className="flex items-center space-x-2 p-2 border rounded">
                  <div className="flex-1 space-y-2">
                    <Input
                      value={attachment.name}
                      onChange={(e) => updateAttachment(attachment.id, 'name', e.target.value)}
                      placeholder="Display name"
                    />
                    <Input
                      value={attachment.path}
                      onChange={(e) => updateAttachment(attachment.id, 'path', e.target.value)}
                      placeholder="File path (e.g., storage/reports/file.pdf)"
                    />
                  </div>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => removeAttachment(attachment.id)}
                  >
                    <Trash2 className="w-4 h-4" />
                  </Button>
                </div>
              ))}
            </div>

            {/* SMTP Configuration */}
            <div className="space-y-4">
              <div className="flex items-center space-x-2">
                <Label htmlFor="useCustomSMTP">Use Custom SMTP</Label>
                <Switch
                  checked={nodeSettings.smtpConfig.useCustom}
                  onCheckedChange={(checked) => updateNodeData('smtpConfig', {
                    ...nodeSettings.smtpConfig,
                    useCustom: checked
                  })}
                />
              </div>

              {nodeSettings.smtpConfig.useCustom && (
                <div className="space-y-3 p-3 border rounded bg-gray-50">
                  <div className="grid grid-cols-2 gap-3">
                    <div className="space-y-2">
                      <Label htmlFor="smtpHost">SMTP Host</Label>
                      <Input
                        value={nodeSettings.smtpConfig.host}
                        onChange={(e) => updateNodeData('smtpConfig', {
                          ...nodeSettings.smtpConfig,
                          host: e.target.value
                        })}
                        placeholder="smtp.gmail.com"
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="smtpPort">Port</Label>
                      <Input
                        type="number"
                        value={nodeSettings.smtpConfig.port}
                        onChange={(e) => updateNodeData('smtpConfig', {
                          ...nodeSettings.smtpConfig,
                          port: parseInt(e.target.value)
                        })}
                        placeholder="587"
                      />
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="smtpEncryption">Encryption</Label>
                    <Select
                      value={nodeSettings.smtpConfig.encryption}
                      onValueChange={(value) => updateNodeData('smtpConfig', {
                        ...nodeSettings.smtpConfig,
                        encryption: value
                      })}
                    >
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="tls">TLS</SelectItem>
                        <SelectItem value="ssl">SSL</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="grid grid-cols-2 gap-3">
                    <div className="space-y-2">
                      <Label htmlFor="smtpUsername">Username</Label>
                      <Input
                        value={nodeSettings.smtpConfig.username}
                        onChange={(e) => updateNodeData('smtpConfig', {
                          ...nodeSettings.smtpConfig,
                          username: e.target.value
                        })}
                        placeholder="your-email@gmail.com"
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="smtpPassword">Password</Label>
                      <Input
                        type="password"
                        value={nodeSettings.smtpConfig.password}
                        onChange={(e) => updateNodeData('smtpConfig', {
                          ...nodeSettings.smtpConfig,
                          password: e.target.value
                        })}
                        placeholder="App password"
                      />
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Response Variable */}
            <div className="space-y-2">
              <Label htmlFor="responseVar">Response Variable</Label>
              <VariableInput
                value={nodeSettings.responseVar}
                onChange={(value) => updateNodeData('responseVar', value)}
                placeholder="Variable name to store email response"
              />
            </div>
          </TabsContent>

          <TabsContent value="setup" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Settings className="w-5 h-5" />
                  <span>Setup Guide</span>
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <Alert>
                  <Info className="h-4 w-4" />
                  <AlertDescription>
                    Follow these steps to set up email sending
                  </AlertDescription>
                </Alert>

                <div className="space-y-4">
                  <div>
                    <h4 className="font-semibold mb-2">Step 1: Configure Global SMTP (Recommended)</h4>
                    <ol className="list-decimal list-inside space-y-1 text-sm">
                      <li>Update your <code className="bg-gray-100 px-1 rounded">.env</code> file with SMTP settings</li>
                      <li>Set <code className="bg-gray-100 px-1 rounded">MAIL_MAILER=smtp</code></li>
                      <li>Configure host, port, username, password</li>
                      <li>This will be used by default unless custom SMTP is enabled</li>
                    </ol>
                  </div>

                  <div>
                    <h4 className="font-semibold mb-2">Step 2: Gmail Setup</h4>
                    <ol className="list-decimal list-inside space-y-1 text-sm">
                      <li>Enable 2-Factor Authentication on your Gmail account</li>
                      <li>Generate an App Password: <a href="https://myaccount.google.com/apppasswords" target="_blank" className="text-blue-600">Google App Passwords</a></li>
                      <li>Use your Gmail address as username</li>
                      <li>Use the generated app password (not your regular password)</li>
                      <li>SMTP Host: <code className="bg-gray-100 px-1 rounded">smtp.gmail.com</code></li>
                      <li>Port: <code className="bg-gray-100 px-1 rounded">587</code> (TLS) or <code className="bg-gray-100 px-1 rounded">465</code> (SSL)</li>
                    </ol>
                  </div>

                  <div>
                    <h4 className="font-semibold mb-2">Step 3: Other Email Providers</h4>
                    <div className="space-y-2">
                      <p className="text-sm font-medium">Outlook/Hotmail:</p>
                      <ul className="list-disc list-inside space-y-1 text-sm ml-4">
                        <li>Host: <code className="bg-gray-100 px-1 rounded">smtp-mail.outlook.com</code></li>
                        <li>Port: <code className="bg-gray-100 px-1 rounded">587</code></li>
                        <li>Encryption: <code className="bg-gray-100 px-1 rounded">TLS</code></li>
                      </ul>
                    </div>
                    <div className="space-y-2 mt-3">
                      <p className="text-sm font-medium">Yahoo:</p>
                      <ul className="list-disc list-inside space-y-1 text-sm ml-4">
                        <li>Host: <code className="bg-gray-100 px-1 rounded">smtp.mail.yahoo.com</code></li>
                        <li>Port: <code className="bg-gray-100 px-1 rounded">587</code></li>
                        <li>Encryption: <code className="bg-gray-100 px-1 rounded">TLS</code></li>
                      </ul>
                    </div>
                  </div>

                  <div>
                    <h4 className="font-semibold mb-2">Step 4: Configure Node</h4>
                    <ol className="list-decimal list-inside space-y-1 text-sm">
                      <li>Set recipient email (use variables like {`{{contact_email}}`})</li>
                      <li>Add CC/BCC recipients if needed</li>
                      <li>Write email subject and body</li>
                      <li>Choose HTML or plain text format</li>
                      <li>Add attachments if needed</li>
                      <li>Enable custom SMTP if using different provider</li>
                      <li>Set response variable name</li>
                    </ol>
                  </div>

                  <div>
                    <h4 className="font-semibold mb-2">Step 5: Variables Available</h4>
                    <div className="text-sm space-y-1">
                      <p><code className="bg-gray-100 px-1 rounded">{`{{contact_name}}`}</code> - Contact's name</p>
                      <p><code className="bg-gray-100 px-1 rounded">{`{{contact_email}}`}</code> - Contact's email</p>
                      <p><code className="bg-gray-100 px-1 rounded">{`{{contact_phone}}`}</code> - Contact's phone</p>
                      <p><code className="bg-gray-100 px-1 rounded">{`{{contact_last_message}}`}</code> - Last message from contact</p>
                      <p><code className="bg-gray-100 px-1 rounded">{`{{variable_name}}`}</code> - Any custom variable from previous nodes</p>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
          </div>

          <Handle
            type="source"
            position={Position.Right}
            style={{ right: '-4px', background: '#555', zIndex: 50 }}
            isConnectable={isConnectable}
          />
        </div>
      </ContextMenuTrigger>
      <ContextMenuContent>
        <ContextMenuItem
          className="text-red-600 focus:text-red-600 focus:bg-red-100"
          onClick={() => deleteNode(id)}
        >
          <Trash2 className="mr-2 h-4 w-4" />
          Delete
        </ContextMenuItem>
      </ContextMenuContent>
    </ContextMenu>
  );
};

export default SendEmailNode;