<?php

namespace Modules\Flowmaker\Models\Nodes;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Modules\Flowmaker\Models\Contact;
// Google API classes will be loaded dynamically

class Sheets extends Node
{
    public function process($message, $data)
    {
        Log::info('🔄 Processing Sheets node', [
            'node_id' => $this->id ?? 'unknown',
            'flow_id' => $this->flow_id ?? 'unknown'
        ]);

        try {
            // Load settings
            $nodeData = $this->getDataAsArray();
            $settings = $nodeData['settings']['sheets'] ?? [];
            Log::debug('📋 Sheets settings', ['settings' => $settings]);

            // Resolve contact
            $contactId = is_object($data) ? ($data->contact_id ?? null) : ($data['contact_id'] ?? null);
            $contact = $contactId ? Contact::find($contactId) : null;
            if (!$contact) {
                Log::error('❌ Contact not found', ['contactId' => $contactId]);
                return ['success' => false];
            }

            // Extract settings
            $operation = $settings['operation'] ?? 'read';
            $spreadsheetId = $settings['spreadsheetId'] ?? '';
            $range = $settings['range'] ?? 'A:Z';
            $authMethod = $settings['authMethod'] ?? 'service_account';
            $responseVar = trim($settings['responseVar'] ?? '');
            $dataToWrite = $settings['dataToWrite'] ?? '';
            $writeMode = $settings['writeMode'] ?? 'append';

            // Interpolate variables
            $spreadsheetId = $contact->changeVariables($spreadsheetId, $this->flow_id);
            $range = $contact->changeVariables($range, $this->flow_id);
            $dataToWrite = $contact->changeVariables($dataToWrite, $this->flow_id);

            if (empty($spreadsheetId)) {
                Log::error('❌ Spreadsheet ID is empty');
                return ['success' => false];
            }

            // Initialize Google Sheets client
            $client = $this->initializeGoogleClient($authMethod, $settings);
            $service = new \Google\Service\Sheets($client);

            $result = null;

            switch ($operation) {
                case 'read':
                    $result = $this->readFromSheet($service, $spreadsheetId, $range);
                    break;
                case 'write':
                    $result = $this->writeToSheet($service, $spreadsheetId, $range, $dataToWrite, $writeMode);
                    break;
                case 'update':
                    $result = $this->updateSheet($service, $spreadsheetId, $range, $dataToWrite);
                    break;
                case 'create_sheet':
                    $result = $this->createSheet($service, $spreadsheetId, $settings['sheetName'] ?? 'New Sheet');
                    break;
                default:
                    throw new \Exception("Unknown operation: {$operation}");
            }

            // Store response in contact state
            if ($responseVar !== '') {
                $contact->setContactState($this->flow_id, $responseVar, json_encode($result));
                Log::info('✅ Stored response in contact state', ['variable' => $responseVar]);
            }

            // Continue to next node
            $nextNode = $this->getNextNodeId();
            if ($nextNode) {
                $nextNode->process($message, $data);
            }

            return ['success' => true, 'data' => $result];

        } catch (\Exception $e) {
            Log::error('❌ Sheets node error', ['error' => $e->getMessage()]);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    private function initializeGoogleClient($authMethod, $settings)
    {
        $client = new \Google\Client();
        $client->setApplicationName('Flowmaker Sheets Integration');
        $client->setScopes([\Google\Service\Sheets::SPREADSHEETS]);

        switch ($authMethod) {
            case 'service_account':
                $credentialsPath = $settings['serviceAccountPath'] ?? storage_path('google-credentials.json');
                if (!file_exists($credentialsPath)) {
                    throw new \Exception('Service account credentials file not found at: ' . $credentialsPath);
                }
                $client->setAuthConfig($credentialsPath);
                break;
            case 'api_key':
                $apiKey = $settings['apiKey'] ?? '';
                if (empty($apiKey)) {
                    throw new \Exception('API key is required for API key authentication');
                }
                $client->setDeveloperKey($apiKey);
                break;
            default:
                throw new \Exception("Unknown authentication method: {$authMethod}");
        }

        return $client;
    }

    private function readFromSheet($service, $spreadsheetId, $range)
    {
        $response = $service->spreadsheets_values->get($spreadsheetId, $range);
        $values = $response->getValues();
        
        return [
            'values' => $values,
            'range' => $range,
            'rowCount' => count($values)
        ];
    }

    private function writeToSheet($service, $spreadsheetId, $range, $dataToWrite, $writeMode)
    {
        // Parse data (JSON, CSV, or simple text)
        $values = $this->parseDataToWrite($dataToWrite);
        
        $body = new \Google\Service\Sheets\ValueRange(['values' => $values]);
        
        $params = [
            'valueInputOption' => 'USER_ENTERED'
        ];

        if ($writeMode === 'append') {
            $response = $service->spreadsheets_values->append($spreadsheetId, $range, $body, $params);
        } else {
            $response = $service->spreadsheets_values->update($spreadsheetId, $range, $body, $params);
        }

        return [
            'updatedRows' => $response->getUpdatedRows(),
            'updatedColumns' => $response->getUpdatedColumns(),
            'updatedCells' => $response->getUpdatedCells()
        ];
    }

    private function parseDataToWrite($dataToWrite)
    {
        // Try to parse as JSON first
        $decoded = json_decode($dataToWrite, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
            return $decoded;
        }

        // Try to parse as CSV
        if (strpos($dataToWrite, ',') !== false) {
            $lines = explode("\n", $dataToWrite);
            $result = [];
            foreach ($lines as $line) {
                if (trim($line)) {
                    $result[] = str_getcsv($line);
                }
            }
            return $result;
        }

        // Treat as simple text (single row)
        return [[$dataToWrite]];
    }

    private function updateSheet($service, $spreadsheetId, $range, $dataToWrite)
    {
        return $this->writeToSheet($service, $spreadsheetId, $range, $dataToWrite, 'update');
    }

    private function createSheet($service, $spreadsheetId, $sheetName)
    {
        $request = new \Google\Service\Sheets\AddSheetRequest();
        $request->setProperties(new \Google\Service\Sheets\SheetProperties());
        $request->getProperties()->setTitle($sheetName);

        $batchUpdateRequest = new \Google\Service\Sheets\BatchUpdateSpreadsheetRequest();
        $batchUpdateRequest->setRequests([$request]);

        $response = $service->spreadsheets->batchUpdate($spreadsheetId, $batchUpdateRequest);
        
        return [
            'sheetId' => $response->getReplies()[0]->getAddSheet()->getProperties()->getSheetId(),
            'sheetName' => $sheetName
        ];
    }

    protected function getNextNodeId($data = null)
    {
        if (!empty($this->outgoingEdges)) {
            return $this->outgoingEdges[0]->getTarget();
        }
        return null;
    }
}