@extends('layouts.app', ['title' => __('crm.create_connection')])

@section('content')
    <div class="header pb-8 pt-5 pt-md-8">
    </div>
    <div class="container-fluid mt--7">
        <div class="row">
            <div class="col">
                <div class="card shadow">
                    <div class="card-header border-0">
                        <div class="row align-items-center">
                            <div class="col-8">
                                <h3 class="mb-0">{{ __('crm.create_connection') }}</h3>
                            </div>
                            <div class="col-4 text-right">
                                <a href="{{ route('contacts.crm-dashboard.index') }}" class="btn btn-sm btn-secondary">
                                    <i class="ni ni-bold-left"></i> {{ __('crm.back_to_connections') }}
                                </a>
                            </div>
                        </div>
                    </div>

                    <div class="col-12">
                        @include('partials.flash')
                    </div>

                    <div class="card-body">
                        <form action="{{ route('contacts.crm.store') }}" method="POST" id="connectionForm">
                            @csrf
                            
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label for="name" class="form-control-label">{{ __('crm.connection_name') }} *</label>
                                        <input type="text" 
                                               name="name" 
                                               id="name" 
                                               class="form-control @error('name') is-invalid @enderror" 
                                               value="{{ old('name') }}" 
                                               required>
                                        @error('name')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label for="type" class="form-control-label">{{ __('crm.crm_type') }} *</label>
                                        <select name="type" 
                                                id="type" 
                                                class="form-control @error('type') is-invalid @enderror" 
                                                required>
                                            <option value="">{{ __('crm.select_crm_type') }}</option>
                                            @foreach($availableTypes as $type => $info)
                                                <option value="{{ $type }}" {{ old('type') == $type ? 'selected' : '' }}>
                                                    {{ $info['display_name'] }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('type')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <div class="row">
                                <div class="col-md-12">
                                    <div class="form-group">
                                        <div class="custom-control custom-checkbox">
                                            <input type="checkbox" 
                                                   name="is_active" 
                                                   id="is_active" 
                                                   class="custom-control-input" 
                                                   value="1" 
                                                   {{ old('is_active', true) ? 'checked' : '' }}>
                                            <label class="custom-control-label" for="is_active">
                                                {{ __('crm.activate_connection') }}
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Dynamic credentials section -->
                            <div id="credentialsSection" style="display: none;">
                                <hr>
                                <h4>{{ __('crm.credentials') }}</h4>
                                
                                <div id="credentialsFields">
                                    <!-- Dynamic fields will be loaded here -->
                                </div>
                                
                                <div class="form-group">
                                    <button type="button" 
                                            id="testConnectionBtn" 
                                            class="btn btn-outline-info" 
                                            disabled>
                                        <i class="ni ni-settings"></i> {{ __('crm.test_connection') }}
                                    </button>
                                    <div id="testResult" class="mt-2"></div>
                                </div>
                            </div>

                            <!-- Field mapping section -->
                            <div id="fieldMappingSection" style="display: none;">
                                <hr>
                                <h4>{{ __('crm.field_mapping') }}</h4>
                                <p class="text-muted">{{ __('crm.field_mapping_description') }}</p>
                                
                                <div id="fieldMappingFields">
                                    <!-- Dynamic field mapping will be loaded here -->
                                </div>
                            </div>

                            <div class="form-group mt-4">
                                <button type="submit" 
                                        id="submitBtn" 
                                        class="btn btn-primary" 
                                        disabled>
                                    <i class="ni ni-check-bold"></i> {{ __('crm.create_connection') }}
                                </button>
                                <a href="{{ route('contacts.crm.index') }}" class="btn btn-secondary">
                                    {{ __('crm.cancel') }}
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        @include('layouts.footers.auth')
    </div>
@endsection

@push('js')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const typeSelect = document.getElementById('type');
    const credentialsSection = document.getElementById('credentialsSection');
    const credentialsFields = document.getElementById('credentialsFields');
    const fieldMappingSection = document.getElementById('fieldMappingSection');
    const fieldMappingFields = document.getElementById('fieldMappingFields');
    const testConnectionBtn = document.getElementById('testConnectionBtn');
    const submitBtn = document.getElementById('submitBtn');
    const testResult = document.getElementById('testResult');

    let availableTypes = @json($availableTypes);

    typeSelect.addEventListener('change', function() {
        const selectedType = this.value;
        
        if (selectedType && availableTypes[selectedType]) {
            loadCredentialsFields(selectedType);
            loadFieldMapping(selectedType);
            credentialsSection.style.display = 'block';
            fieldMappingSection.style.display = 'block';
        } else {
            credentialsSection.style.display = 'none';
            fieldMappingSection.style.display = 'none';
            submitBtn.disabled = true;
        }
    });

    function loadCredentialsFields(type) {
        const typeInfo = availableTypes[type];
        console.log('Type info for', type, ':', typeInfo);
        const requiredCredentials = typeInfo.required_credentials;
        console.log('Required credentials:', requiredCredentials);
        
        credentialsFields.innerHTML = '';
        
        Object.keys(requiredCredentials).forEach(field => {
            const credentialInfo = requiredCredentials[field];
            
            // Handle both string and object formats
            let label, inputType, description;
            if (typeof credentialInfo === 'string') {
                label = credentialInfo;
                inputType = 'text';
                description = '';
            } else if (typeof credentialInfo === 'object') {
                label = credentialInfo.label || field;
                inputType = credentialInfo.type || 'text';
                description = credentialInfo.description || '';
            } else {
                label = field;
                inputType = 'text';
                description = '';
            }
            
            // Determine input type based on field name if not specified
            if (inputType === 'text') {
                if (field.toLowerCase().includes('password') || field.toLowerCase().includes('token')) {
                    inputType = 'password';
                } else if (field.toLowerCase().includes('email')) {
                    inputType = 'email';
                } else if (field.toLowerCase().includes('url')) {
                    inputType = 'url';
                }
            }
            
            // Special handling for Odoo database field
            if (field.toLowerCase() === 'database' && type === 'odoo') {
                description = '{{ __("crm.odoo_database_help") }} <a href="https://your-domain.com/web/database/selector" target="_blank" class="text-primary">{{ __("crm.odoo_database_url") }}</a><br class="text-muted">{{ __("crm.odoo_database_note") }}';
            }
            
            const fieldDiv = document.createElement('div');
            fieldDiv.className = 'form-group';
            
            fieldDiv.innerHTML = `
                <label for="credentials_${field}" class="form-control-label">
                    ${label} *
                </label>
                <input type="${inputType}" 
                       name="credentials[${field}]" 
                       id="credentials_${field}" 
                       class="form-control" 
                       required
                       placeholder="${label}">
                ${description ? `<small class="form-text text-muted">${description}</small>` : ''}
            `;
            
            credentialsFields.appendChild(fieldDiv);
        });
        
        testConnectionBtn.disabled = false;
    }

    function loadFieldMapping(type) {
        // Load field mapping fields via AJAX
        fetch(`{{ route('contacts.crm.fields') }}?type=${type}`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                type: type,
                credentials: getCredentialsData()
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                fieldMappingFields.innerHTML = '';
                
                const defaultMappings = {
                    'name': 'firstname',
                    'lastname': 'lastname',
                    'email': 'email',
                    'phone': 'phone',
                    'company': 'company'
                };
                
                Object.keys(defaultMappings).forEach(localField => {
                    const fieldDiv = document.createElement('div');
                    fieldDiv.className = 'form-group col-md-6';
                    
                    fieldDiv.innerHTML = `
                        <label for="field_mapping_${localField}" class="form-control-label">
                            ${localField.charAt(0).toUpperCase() + localField.slice(1)}
                        </label>
                        <select name="field_mapping[${localField}]" 
                                id="field_mapping_${localField}" 
                                class="form-control">
                            <option value="">{{ __('crm.select_field') }}</option>
                            ${data.fields.map(field => 
                                `<option value="${field.name}" ${field.name === defaultMappings[localField] ? 'selected' : ''}>
                                    ${field.label} (${field.name})
                                </option>`
                            ).join('')}
                        </select>
                    `;
                    
                    fieldMappingFields.appendChild(fieldDiv);
                });
            }
        })
        .catch(error => {
            console.error('Error loading field mapping:', error);
        });
    }

    testConnectionBtn.addEventListener('click', function() {
        const type = typeSelect.value;
        const credentials = getCredentialsData();
        
        if (!type || !credentials) {
            return;
        }
        
        testConnectionBtn.disabled = true;
        testResult.innerHTML = '<div class="spinner-border spinner-border-sm" role="status"></div> Testing connection...';
        
        fetch(`{{ route('contacts.crm.test') }}`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                type: type,
                credentials: credentials
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                testResult.innerHTML = '<div class="alert alert-success"><i class="ni ni-check-bold"></i> ' + data.message + '</div>';
                submitBtn.disabled = false;
            } else {
                testResult.innerHTML = '<div class="alert alert-danger"><i class="ni ni-fat-remove"></i> ' + data.message + '</div>';
                submitBtn.disabled = true;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            testResult.innerHTML = '<div class="alert alert-danger"><i class="ni ni-fat-remove"></i> Connection test failed</div>';
            submitBtn.disabled = true;
        })
        .finally(() => {
            testConnectionBtn.disabled = false;
        });
    });

    function getCredentialsData() {
        const credentials = {};
        const credentialInputs = document.querySelectorAll('input[name^="credentials["]');
        
        credentialInputs.forEach(input => {
            const fieldName = input.name.match(/credentials\[(.*?)\]/)[1];
            credentials[fieldName] = input.value;
        });
        
        return credentials;
    }
});
</script>
@endpush