<?php

namespace Modules\Contacts\Providers;

use Illuminate\Support\Facades\Blade;
use Illuminate\Support\ServiceProvider as Provider;

class Main extends Provider
{
    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register()
    {
        $this->loadConfig();
        $this->loadRoutes();
        $this->registerServices();
    }

    /**
     * Boot the application events.
     *
     * @return void
     */
    public function boot()
    {
        $this->publishConfig();
        $this->loadViews();
        $this->loadViewComponents();
        $this->loadTranslations();
        $this->loadMigrations();
    }

    /**
     * Load config.
     *
     * @return void
     */
    protected function loadConfig()
    {
        $this->mergeConfigFrom(
            __DIR__ . '/../Config/config.php', 'contacts'
        );
    }

    /**
     * Publish config.
     *
     * @return void
     */
    protected function publishConfig()
    {
        $this->publishes([
            __DIR__ . '/../Config/config.php' => config_path('contacts.php'),
        ], 'config');
    }

    /**
     * Load views.
     *
     * @return void
     */
    public function loadViews()
    {
        $viewPath = resource_path('views/modules/contacts');

        $sourcePath = __DIR__ . '/../Resources/views';

        $this->publishes([
            $sourcePath => $viewPath
        ],'views');

        $this->loadViewsFrom(array_merge(array_map(function ($path) {
            return $path . '/modules/contacts';
        }, \Config::get('view.paths')), [$sourcePath]), 'contacts');
    }

    /**
     * Load view components.
     *
     * @return void
     */
    public function loadViewComponents()
    {
        Blade::componentNamespace('Modules\Contacts\View\Components', 'contacts');
    }

    /**
     * Load translations.
     *
     * @return void
     */
    public function loadTranslations()
    {
        $langPath = resource_path('lang/modules/contacts');

        if (is_dir($langPath)) {
            $this->loadTranslationsFrom($langPath, 'contacts');
        } else {
            $this->loadTranslationsFrom(__DIR__ . '/../Resources/lang/en', 'contacts');
        }
    }

    /**
     * Load migrations.
     *
     * @return void
     */
    public function loadMigrations()
    {
        $this->loadMigrationsFrom(__DIR__ . '/../Database/Migrations');
    }

    /**
     * Load routes.
     *
     * @return void
     */
    public function loadRoutes()
    {
        if (app()->routesAreCached()) {
            return;
        }

        $routes = [
            'web.php',
            'api.php',
        ];

        foreach ($routes as $route) {
            $this->loadRoutesFrom(__DIR__ . '/../Routes/' . $route);
        }
    }

    /**
     * Register CRM services.
     *
     * @return void
     */
    protected function registerServices()
    {
        // Register CRM Plugin Registry as singleton
        $this->app->singleton(\Modules\Contacts\Services\CrmIntegration\CrmPluginRegistry::class, function ($app) {
            return new \Modules\Contacts\Services\CrmIntegration\CrmPluginRegistry();
        });

        // Register CRM Import Service
        $this->app->bind(\Modules\Contacts\Services\EnhancedCrmImportService::class, function ($app) {
            return new \Modules\Contacts\Services\EnhancedCrmImportService(
                $app->make(\Modules\Contacts\Services\CrmIntegration\CrmPluginRegistry::class),
                $app->make(\Modules\Contacts\Services\DuplicateDetectionService::class),
                $app->make(\Modules\Contacts\Services\PhoneNormalizationService::class)
            );
        });

        // Register Duplicate Detection Service
        $this->app->singleton(\Modules\Contacts\Services\DuplicateDetectionService::class, function ($app) {
            return new \Modules\Contacts\Services\DuplicateDetectionService(
                $app->make(\Modules\Contacts\Services\PhoneNormalizationService::class)
            );
        });

        // Register Phone Normalization Service
        $this->app->singleton(\Modules\Contacts\Services\PhoneNormalizationService::class, function ($app) {
            return new \Modules\Contacts\Services\PhoneNormalizationService();
        });

        // Register HubSpot Connector
        $this->app->bind(\Modules\Contacts\Services\CrmIntegration\Connectors\HubSpotConnector::class, function ($app) {
            return new \Modules\Contacts\Services\CrmIntegration\Connectors\HubSpotConnector();
        });
    }

    /**
     * Get the services provided by the provider.
     *
     * @return array
     */
    public function provides()
    {
        return [
            \Modules\Contacts\Services\CrmIntegration\CrmPluginRegistry::class,
            \Modules\Contacts\Services\EnhancedCrmImportService::class,
            \Modules\Contacts\Services\DuplicateDetectionService::class,
            \Modules\Contacts\Services\PhoneNormalizationService::class,
            \Modules\Contacts\Services\CrmIntegration\Connectors\HubSpotConnector::class,
        ];
    }
}
