<?php

namespace Modules\Contacts\Models;

use App\Models\Company;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Scopes\CompanyScope;

class CrmSyncHistory extends Model
{
    use SoftDeletes;

    protected $table = 'crm_sync_history';
    public $guarded = [];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'import_settings' => 'array',
        'field_mapping' => 'array',
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
    ];

    /**
     * Boot the model.
     */
    protected static function booted()
    {
        static::addGlobalScope(new CompanyScope);

        static::creating(function ($model) {
            $company_id = session('company_id', null);
            
            // If no company_id in session, try to get it from the authenticated user
            if (!$company_id && auth()->check() && auth()->user()->company_id) {
                $company_id = auth()->user()->company_id;
            }
            
            // If still no company_id, try to get the first company (for testing purposes)
            if (!$company_id) {
                $company = \App\Models\Company::first();
                if ($company) {
                    $company_id = $company->id;
                }
            }
            
            if ($company_id) {
                $model->company_id = $company_id;
            }
        });
    }

    /**
     * Get the company that owns the sync history.
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the CRM connection for this sync history.
     */
    public function crmConnection()
    {
        return $this->belongsTo(CrmConnection::class);
    }

    /**
     * Check if the sync is completed.
     */
    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }

    /**
     * Check if the sync is in progress.
     */
    public function isInProgress(): bool
    {
        return $this->status === 'in_progress';
    }

    /**
     * Check if the sync has failed.
     */
    public function hasFailed(): bool
    {
        return $this->status === 'failed';
    }

    /**
     * Check if the sync is pending.
     */
    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    /**
     * Check if the sync is cancelled.
     */
    public function isCancelled(): bool
    {
        return $this->status === 'cancelled';
    }

    /**
     * Get the sync status display name.
     */
    public function getStatusDisplayAttribute(): string
    {
        return ucfirst(str_replace('_', ' ', $this->status));
    }

    /**
     * Get the sync type display name.
     */
    public function getSyncTypeDisplayAttribute(): string
    {
        return ucfirst($this->sync_type);
    }

    /**
     * Get the progress percentage.
     */
    public function getProgressPercentageAttribute(): float
    {
        if ($this->total_records === 0) {
            return 0;
        }

        return round(($this->processed_records / $this->total_records) * 100, 2);
    }

    /**
     * Get the duration in seconds.
     */
    public function getDurationAttribute(): ?int
    {
        if (!$this->started_at || !$this->completed_at) {
            return null;
        }

        return $this->completed_at->diffInSeconds($this->started_at);
    }

    /**
     * Get the duration in human readable format.
     */
    public function getDurationHumanAttribute(): ?string
    {
        $duration = $this->duration;
        if (!$duration) {
            return null;
        }

        if ($duration < 60) {
            return $duration . ' seconds';
        } elseif ($duration < 3600) {
            return round($duration / 60) . ' minutes';
        } else {
            return round($duration / 3600) . ' hours';
        }
    }

    /**
     * Scope to get only completed syncs.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope to get only failed syncs.
     */
    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    /**
     * Scope to get only in progress syncs.
     */
    public function scopeInProgress($query)
    {
        return $query->where('status', 'in_progress');
    }

    /**
     * Scope to get only pending syncs.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope to get syncs by type.
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('sync_type', $type);
    }

    /**
     * Scope to get recent syncs.
     */
    public function scopeRecent($query, $days = 7)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }
}