<?php

namespace Modules\Contacts\Http\Controllers;

use Modules\Contacts\Models\CrmConnection;
use Modules\Contacts\Services\CrmIntegration\CrmPluginRegistry;
use Modules\Contacts\Http\Requests\StoreCrmConnectionRequest;
use Modules\Contacts\Http\Requests\UpdateCrmConnectionRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;

class CrmConnectionController extends Controller
{
    protected $crmRegistry;

    public function __construct(CrmPluginRegistry $crmRegistry)
    {
        $this->crmRegistry = $crmRegistry;
    }

    /**
     * Display a listing of CRM connections.
     * Now directly shows the comprehensive CRM dashboard.
     */
    public function index()
    {
        return $this->dashboard();
    }

    /**
     * Show the form for creating a new CRM connection.
     */
    public function create()
    {
        $this->authChecker();
        
        $availableTypes = $this->crmRegistry->getAllConnectorInfo();
        
        return view('contacts::crm.create', compact('availableTypes'));
    }

    /**
     * Store a newly created CRM connection.
     */
    public function store(StoreCrmConnectionRequest $request)
    {
        $this->authChecker();

        try {
            $connector = $this->crmRegistry->get($request->type);
            
            // Test connection before saving
            if (!$connector->testConnection($request->credentials)) {
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['credentials' => __('crm.connection_test_failed')]);
            }

            $connection = CrmConnection::create([
                'name' => $request->name,
                'type' => $request->type,
                'credentials' => $request->credentials,
                'field_mapping' => $request->field_mapping ?? [],
                'is_active' => $request->boolean('is_active', true),
                'is_healthy' => true,
                'last_test_at' => now(),
            ]);

            return redirect()->route('contacts.crm.index')
                ->withStatus(__('crm.connection_created_successfully'));

        } catch (\Exception $e) {
            Log::error('Failed to create CRM connection', [
                'error' => $e->getMessage(),
                'type' => $request->type,
            ]);

            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => __('crm.connection_creation_failed')]);
        }
    }

    /**
     * Display the specified CRM connection.
     */
    public function show(CrmConnection $crmConnection)
    {
        $this->authChecker();
        
        $crmConnection->load(['syncHistory' => function ($query) {
            $query->orderBy('created_at', 'desc')->limit(10);
        }]);

        return view('contacts::crm.show', compact('crmConnection'));
    }

    /**
     * Show the form for editing the specified CRM connection.
     */
    public function edit(CrmConnection $crmConnection)
    {
        $this->authChecker();
        
        $availableTypes = $this->crmRegistry->getAllConnectorInfo();
        
        return view('contacts::crm.edit', compact('crmConnection', 'availableTypes'));
    }

    /**
     * Update the specified CRM connection.
     */
    public function update(UpdateCrmConnectionRequest $request, CrmConnection $crmConnection)
    {
        $this->authChecker();

        try {
            $connector = $this->crmRegistry->get($request->type);
            
            // Test connection before updating
            if (!$connector->testConnection($request->credentials)) {
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['credentials' => __('crm.connection_test_failed')]);
            }

            $crmConnection->update([
                'name' => $request->name,
                'type' => $request->type,
                'credentials' => $request->credentials,
                'field_mapping' => $request->field_mapping ?? $crmConnection->field_mapping,
                'is_active' => $request->boolean('is_active', true),
                'is_healthy' => true,
                'last_test_at' => now(),
            ]);

            return redirect()->route('contacts.crm.index')
                ->withStatus(__('crm.connection_updated_successfully'));

        } catch (\Exception $e) {
            Log::error('Failed to update CRM connection', [
                'error' => $e->getMessage(),
                'connection_id' => $crmConnection->id,
            ]);

            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => __('crm.connection_update_failed')]);
        }
    }

    /**
     * Remove the specified CRM connection.
     */
    public function destroy(CrmConnection $crmConnection)
    {
        $this->authChecker();

        try {
            $crmConnection->delete();

            // Check if this is an AJAX request
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => __('crm.connection_deleted_successfully'),
                ]);
            }

            return redirect()->route('contacts.crm.index')
                ->withStatus(__('crm.connection_deleted_successfully'));

        } catch (\Exception $e) {
            Log::error('Failed to delete CRM connection', [
                'error' => $e->getMessage(),
                'connection_id' => $crmConnection->id,
            ]);

            // Check if this is an AJAX request
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => __('crm.connection_deletion_failed'),
                ], 500);
            }

            return redirect()->back()
                ->withErrors(['error' => __('crm.connection_deletion_failed')]);
        }
    }

    /**
     * Test a CRM connection.
     */
    public function test(Request $request)
    {
        $this->authChecker();

        $request->validate([
            'type' => 'required|string',
            'credentials' => 'required|array',
        ]);

        try {
            $connector = $this->crmRegistry->get($request->type);
            
            if ($connector->testConnection($request->credentials)) {
                return response()->json([
                    'success' => true,
                    'message' => __('crm.connection_test_successful'),
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => __('crm.connection_test_failed'),
                ], 400);
            }

        } catch (\Exception $e) {
            Log::error('CRM connection test failed', [
                'error' => $e->getMessage(),
                'type' => $request->type,
            ]);

            return response()->json([
                'success' => false,
                'message' => __('crm.connection_test_error'),
            ], 500);
        }
    }

    /**
     * Get available fields for a CRM type.
     */
    public function getFields(Request $request)
    {
        $this->authChecker();

        $request->validate([
            'type' => 'required|string',
            'credentials' => 'required|array',
        ]);

        try {
            $connector = $this->crmRegistry->get($request->type);
            $fields = $connector->getAvailableFields($request->credentials);

            return response()->json([
                'success' => true,
                'fields' => $fields,
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to get CRM fields', [
                'error' => $e->getMessage(),
                'type' => $request->type,
            ]);

            return response()->json([
                'success' => false,
                'message' => __('crm.fields_fetch_failed'),
            ], 500);
        }
    }

    /**
     * Toggle connection status.
     */
    public function toggleStatus(CrmConnection $crmConnection)
    {
        $this->authChecker();

        try {
            $crmConnection->update([
                'is_active' => !$crmConnection->is_active,
            ]);

            $status = $crmConnection->is_active ? 'activated' : 'deactivated';

            return response()->json([
                'success' => true,
                'message' => __("crm.connection_{$status}_successfully"),
                'is_active' => $crmConnection->is_active,
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to toggle CRM connection status', [
                'error' => $e->getMessage(),
                'connection_id' => $crmConnection->id,
            ]);

            return response()->json([
                'success' => false,
                'message' => __('crm.status_toggle_failed'),
            ], 500);
        }
    }

    /**
     * Display the CRM dashboard with comprehensive management interface.
     *
     * @return \Illuminate\View\View
     */
    public function dashboard()
    {
        $this->authChecker();
        
        // Get all CRM connections with pagination
        $connections = CrmConnection::with(['syncHistory' => function($query) {
            $query->latest()->limit(5);
        }])->paginate(10);
        
        // Get recent import history
        $recentImports = \Modules\Contacts\Models\CrmSyncHistory::with('crmConnection')
            ->latest()
            ->limit(10)
            ->get();
        
        // Get statistics
        $stats = [
            'total_connections' => CrmConnection::count(),
            'active_connections' => CrmConnection::where('is_active', true)->count(),
            'healthy_connections' => CrmConnection::where('is_healthy', true)->count(),
            'total_imports' => \Modules\Contacts\Models\CrmSyncHistory::count(),
            'successful_imports' => \Modules\Contacts\Models\CrmSyncHistory::where('status', 'completed')->count(),
            'failed_imports' => \Modules\Contacts\Models\CrmSyncHistory::where('status', 'failed')->count(),
        ];
        
        return view('contacts::crm.dashboard', compact('connections', 'recentImports', 'stats'));
    }

    /**
     * Check authorization.
     */
    private function authChecker()
    {
        // Implement your authorization logic here
        // This should check if the user has permission to manage CRM connections
        if (!auth()->check()) {
            abort(401);
        }
    }
}