<?php

namespace Modules\Contacts\Services\CrmIntegration\Connectors;

use Modules\Contacts\Services\CrmIntegration\CrmConnectorInterface;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

class PipedriveConnector implements CrmConnectorInterface
{
    private const API_BASE_URL = 'https://api.pipedrive.com/v1';

    public function __construct()
    {
        // No credentials stored in constructor
    }

    /**
     * Test the CRM connection
     */
    public function testConnection(array $credentials): bool
    {
        try {
            $response = Http::get(self::API_BASE_URL . '/users/me', [
                'api_token' => $credentials['api_token']
            ]);

            return $response->successful();

        } catch (Exception $e) {
            Log::error('Pipedrive connection test failed', [
                'error' => $e->getMessage(),
                'credentials' => $credentials
            ]);

            return false;
        }
    }

    /**
     * Get contacts from Pipedrive
     */
    public function getContacts(array $credentials, array $options = []): array
    {
        try {
            $limit = min($options['limit'] ?? 100, 500); // Pipedrive limit
            $start = $options['offset'] ?? 0;

            $response = Http::get(self::API_BASE_URL . '/persons', [
                'api_token' => $credentials['api_token'],
                'limit' => $limit,
                'start' => $start,
                'sort' => 'update_time DESC'
            ]);

            if ($response->successful()) {
                $data = $response->json();
                return [
                    'success' => true,
                    'contacts' => $data['data'] ?? [],
                    'total' => $data['additional_data']['pagination']['total_count'] ?? 0,
                    'has_more' => $data['additional_data']['pagination']['more_items_in_collection'] ?? false
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to fetch Pipedrive contacts: ' . $response->body()
            ];

        } catch (Exception $e) {
            Log::error('Pipedrive contacts fetch failed', [
                'error' => $e->getMessage(),
                'options' => $options
            ]);

            return [
                'success' => false,
                'message' => 'Failed to fetch Pipedrive contacts: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get a single contact from Pipedrive
     */
    public function getContact(array $credentials, string $contactId): ?array
    {
        try {
            $response = Http::get(self::API_BASE_URL . '/persons/' . $contactId, [
                'api_token' => $credentials['api_token']
            ]);

            if ($response->successful()) {
                return $response->json();
            }

            return null;

        } catch (Exception $e) {
            Log::error('Pipedrive contact fetch failed', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId
            ]);

            return null;
        }
    }

    /**
     * Create a contact in Pipedrive
     */
    public function createContact(array $credentials, array $contactData): array
    {
        try {
            $response = Http::post(self::API_BASE_URL . '/persons', [
                'api_token' => $credentials['api_token'],
                'name' => $contactData['name'] ?? '',
                'email' => $contactData['email'] ?? null,
                'phone' => $contactData['phone'] ?? null,
                'org_name' => $contactData['company'] ?? null
            ]);

            if ($response->successful()) {
                return $response->json();
            }

            throw new Exception('Failed to create Pipedrive contact: ' . $response->body());

        } catch (Exception $e) {
            Log::error('Pipedrive contact creation failed', [
                'error' => $e->getMessage(),
                'contact_data' => $contactData
            ]);

            throw $e;
        }
    }

    /**
     * Update a contact in Pipedrive
     */
    public function updateContact(array $credentials, string $contactId, array $contactData): array
    {
        try {
            $response = Http::put(self::API_BASE_URL . '/persons/' . $contactId, [
                'api_token' => $credentials['api_token'],
                'name' => $contactData['name'] ?? '',
                'email' => $contactData['email'] ?? null,
                'phone' => $contactData['phone'] ?? null,
                'org_name' => $contactData['company'] ?? null
            ]);

            if ($response->successful()) {
                return $response->json();
            }

            throw new Exception('Failed to update Pipedrive contact: ' . $response->body());

        } catch (Exception $e) {
            Log::error('Pipedrive contact update failed', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId,
                'contact_data' => $contactData
            ]);

            throw $e;
        }
    }

    /**
     * Delete a contact from Pipedrive
     */
    public function deleteContact(array $credentials, string $contactId): bool
    {
        try {
            $response = Http::delete(self::API_BASE_URL . '/persons/' . $contactId, [
                'api_token' => $credentials['api_token']
            ]);

            return $response->successful();

        } catch (Exception $e) {
            Log::error('Pipedrive contact deletion failed', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId
            ]);

            return false;
        }
    }

    /**
     * Get available fields from Pipedrive
     */
    public function getAvailableFields(array $credentials): array
    {
        try {
            $response = Http::get(self::API_BASE_URL . '/personFields', [
                'api_token' => $credentials['api_token']
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $fields = [];

                // Define essential fields for contact mapping
                $essentialFields = [
                    'first_name', 'last_name', 'email', 'phone', 'mobile', 'company_name',
                    'job_title', 'department', 'industry', 'website', 'address',
                    'city', 'state', 'country', 'postal_code', 'timezone',
                    'created_time', 'update_time', 'last_activity_date', 'last_contacted_date',
                    'notes', 'status', 'lead_source'
                ];

                foreach ($data['data'] ?? [] as $field) {
                    // Only include essential fields to avoid overwhelming the dropdown
                    if (in_array($field['key'], $essentialFields)) {
                        $fields[] = [
                            'name' => $field['key'],
                            'label' => $field['name'],
                            'type' => $field['field_type'],
                            'required' => $field['mandatory_flag'] ?? false,
                            'updateable' => $field['edit_flag'] ?? false
                        ];
                    }
                }

                return $fields;
            }

            throw new Exception('Failed to fetch Pipedrive fields: ' . $response->body());

        } catch (Exception $e) {
            Log::error('Pipedrive fields fetch failed', [
                'error' => $e->getMessage()
            ]);

            throw $e;
        }
    }

    /**
     * Transform Pipedrive contact data to our system format
     */
    public function transformContactData(array $crmContact): array
    {
        // Extract name from first_name and last_name
        $firstName = $crmContact['first_name'] ?? '';
        $lastName = $crmContact['last_name'] ?? '';
        $name = trim($firstName . ' ' . $lastName);

        // Extract email from primary email
        $email = null;
        if (isset($crmContact['primary_email'])) {
            $email = $crmContact['primary_email'];
        } elseif (isset($crmContact['email'])) {
            $email = is_array($crmContact['email']) ? $crmContact['email'][0]['value'] : $crmContact['email'];
        }

        // Extract phone from primary phone
        $phone = null;
        if (isset($crmContact['primary_phone'])) {
            $phone = $crmContact['primary_phone'];
        } elseif (isset($crmContact['phone'])) {
            $phone = is_array($crmContact['phone']) ? $crmContact['phone'][0]['value'] : $crmContact['phone'];
        }

        return [
            'name' => $name ?: 'Unknown Contact',
            'email' => $email,
            'phone' => $phone,
            'company' => $crmContact['org_name'] ?? null,
            'crm_id' => $crmContact['id'] ?? null,
            'crm_data' => $crmContact,
            'created_at' => $crmContact['add_time'] ?? now(),
            'updated_at' => $crmContact['update_time'] ?? now()
        ];
    }

    /**
     * Get CRM type
     */
    public function getType(): string
    {
        return 'pipedrive';
    }

    /**
     * Get CRM display name
     */
    public function getDisplayName(): string
    {
        return 'Pipedrive';
    }

    /**
     * Get required credentials
     */
    public function getRequiredCredentials(): array
    {
        return [
            'api_token' => 'API Token',
            'company_domain' => 'Company Domain (optional)'
        ];
    }

    /**
     * Validate credentials
     */
    public function validateCredentials(array $credentials): bool
    {
        return !empty($credentials['api_token']);
    }
}