<?php

namespace Modules\Contacts\Services\CrmIntegration\Connectors;

use Modules\Contacts\Services\CrmIntegration\CrmConnectorInterface;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

class OdooConnector implements CrmConnectorInterface
{
    private $baseUrl;

    public function __construct()
    {
        // No credentials stored in constructor
    }

    /**
     * Authenticate with Odoo and get session info
     */
    private function authenticate(array $credentials): ?array
    {
        try {
            // Use the JSON-RPC authentication method for modern Odoo
            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ])->post($credentials['url'] . '/web/session/authenticate', [
                'jsonrpc' => '2.0',
                'method' => 'call',
                'params' => [
                    'db' => $credentials['database'],
                    'login' => $credentials['username'],
                    'password' => $credentials['password']
                ],
                'id' => 1
            ]);

            if ($response->successful()) {
                $data = $response->json();
                
                if (isset($data['result']) && is_array($data['result']) && isset($data['result']['uid']) && $data['result']['uid']) {
                    // Extract session ID from response cookies
                    $cookies = $response->cookies();
                    $sessionId = null;
                    
                    foreach ($cookies as $cookie) {
                        if ($cookie->getName() === 'session_id') {
                            $sessionId = $cookie->getValue();
                            break;
                        }
                    }
                    
                    if ($sessionId) {
                        Log::info('Odoo authentication successful', [
                            'url' => $credentials['url'],
                            'database' => $credentials['database'],
                            'username' => $credentials['username'],
                            'uid' => $data['result']['uid']
                        ]);
                        return [
                            'session_id' => $sessionId,
                            'uid' => $data['result']['uid'],
                            'user_context' => is_array($data['result']) ? ($data['result']['user_context'] ?? []) : []
                        ];
                    }
                }
            }

            Log::error('Odoo authentication failed', [
                'url' => $credentials['url'],
                'status' => $response->status(),
                'body' => $response->body(),
                'response' => $response->json()
            ]);

            return null;

        } catch (Exception $e) {
            Log::error('Odoo authentication failed', [
                'error' => $e->getMessage(),
                'url' => $credentials['url']
            ]);
            return null;
        }
    }

    /**
     * Make authenticated request to Odoo
     */
    private function makeRequest(array $credentials, string $model, string $method, array $params = []): array
    {
        $authInfo = $this->authenticate($credentials);
        if (!$authInfo) {
            throw new Exception('Failed to authenticate with Odoo');
        }

        $response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'Cookie' => 'session_id=' . $authInfo['session_id']
        ])->post($credentials['url'] . '/web/dataset/call_kw', [
            'jsonrpc' => '2.0',
            'method' => 'call',
            'params' => [
                'model' => $model,
                'method' => $method,
                'args' => $params['args'] ?? [],
                'kwargs' => $params['kwargs'] ?? []
            ],
            'id' => 1
        ]);

        if ($response->successful()) {
            $data = $response->json();
            
            if (isset($data['result'])) {
                // Handle different return types
                if (is_array($data['result'])) {
                    return $data['result'];
                } else {
                    // For scalar values like search_count
                    return [$data['result']];
                }
            }
        }

        Log::error('Odoo API request failed', [
            'model' => $model,
            'method' => $method,
            'status' => $response->status(),
            'body' => $response->body(),
            'response' => $response->json()
        ]);

        throw new Exception('Odoo API request failed: ' . $response->body());
    }

    /**
     * Test the CRM connection
     */
    public function testConnection(array $credentials): bool
    {
        try {
            // Test authentication by making a simple request
            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ])->post($credentials['url'] . '/web/session/authenticate', [
                'jsonrpc' => '2.0',
                'method' => 'call',
                'params' => [
                    'db' => $credentials['database'],
                    'login' => $credentials['username'],
                    'password' => $credentials['password']
                ],
                'id' => 1
            ]);

            if ($response->successful()) {
                $data = $response->json();
                
                if (isset($data['result']['uid']) && $data['result']['uid']) {
                    Log::info('Odoo connection test successful', [
                        'url' => $credentials['url'],
                        'database' => $credentials['database'],
                        'username' => $credentials['username'],
                        'uid' => $data['result']['uid']
                    ]);
                    
                    return true;
                }
            }

            Log::error('Odoo connection test failed', [
                'url' => $credentials['url'],
                'status' => $response->status(),
                'body' => $response->body(),
                'response' => $response->json()
            ]);

            return false;

        } catch (Exception $e) {
            Log::error('Odoo connection test failed', [
                'error' => $e->getMessage(),
                'url' => $credentials['url'],
                'database' => $credentials['database']
            ]);

            return false;
        }
    }

    /**
     * Get contacts from Odoo
     */
    public function getContacts(array $credentials, array $options = []): array
    {
        try {
            $limit = min($options['limit'] ?? 100, 1000); // Odoo limit
            $offset = $options['offset'] ?? 0;
            
            // Determine contact type filter based on options
            $contactType = $options['contact_type'] ?? 'both'; // 'individual', 'company', or 'both'
            $domain = $this->buildContactDomain($contactType);

            $contacts = $this->makeRequest($credentials, 'res.partner', 'search_read', [
                'args' => [
                    $domain,
                    [
                        'id', 'name', 'email', 'phone', 'mobile', 'street', 'city', 
                        'state_id', 'country_id', 'zip', 'website', 'comment',
                        'create_date', 'write_date', 'parent_id', 'category_id',
                        'is_company', 'title', 'function', 'lang', 'tz', 'user_id', 'vat'
                    ],
                    $offset,
                    $limit,
                    'write_date desc'
                ],
                'kwargs' => []
            ]);

            // Get total count
            $totalCountResult = $this->makeRequest($credentials, 'res.partner', 'search_count', [
                'args' => [$domain],
                'kwargs' => []
            ]);
            
            $totalCount = is_array($totalCountResult) ? ($totalCountResult[0] ?? 0) : $totalCountResult;

            // Ensure contacts is an array
            $contactsArray = is_array($contacts) ? $contacts : [];
            
            return [
                'success' => true,
                'contacts' => $contactsArray,
                'total' => $totalCount,
                'has_more' => count($contactsArray) >= $limit,
                'contact_type' => $contactType
            ];

        } catch (Exception $e) {
            Log::error('Odoo contacts fetch failed', [
                'error' => $e->getMessage(),
                'options' => $options
            ]);

            return [
                'success' => false,
                'message' => 'Failed to fetch Odoo contacts: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Build domain filter based on contact type
     */
    private function buildContactDomain(string $contactType): array
    {
        switch ($contactType) {
            case 'individual':
                return [['is_company', '=', false]];
            case 'company':
                return [['is_company', '=', true]];
            case 'both':
            default:
                return []; // No filter - get all contacts
        }
    }

    /**
     * Get a single contact from Odoo
     */
    public function getContact(array $credentials, string $contactId): ?array
    {
        try {
            $contact = $this->makeRequest($credentials, 'res.partner', 'read', [
                'args' => [$contactId],
                'kwargs' => ['fields' => [
                    'id', 'name', 'email', 'phone', 'mobile', 'street', 'city', 
                    'state_id', 'country_id', 'zip', 'website', 'comment',
                    'create_date', 'write_date', 'parent_id', 'category_id'
                ]]
            ]);

            return $contact[0] ?? null;

        } catch (Exception $e) {
            Log::error('Odoo contact fetch failed', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId
            ]);

            return null;
        }
    }

    /**
     * Create a contact in Odoo
     */
    public function createContact(array $credentials, array $contactData): array
    {
        try {
            $contactId = $this->makeRequest($credentials, 'res.partner', 'create', [
                'args' => [[
                    'name' => $contactData['name'] ?? '',
                    'email' => $contactData['email'] ?? null,
                    'phone' => $contactData['phone'] ?? null,
                    'is_company' => false
                ]],
                'kwargs' => []
            ]);

            return ['id' => $contactId];

        } catch (Exception $e) {
            Log::error('Odoo contact creation failed', [
                'error' => $e->getMessage(),
                'contact_data' => $contactData
            ]);

            throw $e;
        }
    }

    /**
     * Update a contact in Odoo
     */
    public function updateContact(array $credentials, string $contactId, array $contactData): array
    {
        try {
            $this->makeRequest($credentials, 'res.partner', 'write', [
                'args' => [$contactId, [
                    'name' => $contactData['name'] ?? '',
                    'email' => $contactData['email'] ?? null,
                    'phone' => $contactData['phone'] ?? null,
                ]],
                'kwargs' => []
            ]);

            return ['id' => $contactId];

        } catch (Exception $e) {
            Log::error('Odoo contact update failed', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId,
                'contact_data' => $contactData
            ]);

            throw $e;
        }
    }

    /**
     * Delete a contact from Odoo
     */
    public function deleteContact(array $credentials, string $contactId): bool
    {
        try {
            $this->makeRequest($credentials, 'res.partner', 'unlink', [
                'args' => [$contactId],
                'kwargs' => []
            ]);

            return true;

        } catch (Exception $e) {
            Log::error('Odoo contact deletion failed', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId
            ]);

            return false;
        }
    }

    /**
     * Get available fields from Odoo
     */
    public function getAvailableFields(array $credentials): array
    {
        try {
            $fields = $this->makeRequest($credentials, 'res.partner', 'fields_get', [
                'args' => [],
                'kwargs' => []
            ]);

            $fieldList = [];
            foreach ($fields as $fieldName => $fieldInfo) {
                // Only include relevant contact fields
                if (in_array($fieldName, [
                    'name', 'email', 'phone', 'mobile', 'street', 'city', 'state_id',
                    'country_id', 'zip', 'website', 'comment', 'parent_id', 'category_id',
                    'title', 'function', 'lang', 'tz', 'user_id', 'vat', 'is_company'
                ])) {
                    $fieldList[] = [
                        'name' => $fieldName,
                        'label' => $fieldInfo['string'] ?? $fieldName,
                        'type' => $fieldInfo['type'] ?? 'char',
                        'required' => $fieldInfo['required'] ?? false,
                        'updateable' => !($fieldInfo['readonly'] ?? false)
                    ];
                }
            }

            return $fieldList;

        } catch (Exception $e) {
            Log::error('Odoo fields fetch failed', [
                'error' => $e->getMessage()
            ]);

            throw $e;
        }
    }

    /**
     * Transform Odoo contact data to our system format
     */
    public function transformContactData(array $crmContact): array
    {
        // Extract name
        $name = $crmContact['name'] ?? 'Unknown Contact';
        
        // Check if this is a company contact
        $isCompany = $crmContact['is_company'] ?? false;

        // Extract email (prefer email over mobile for contact info)
        $email = $crmContact['email'] ?? null;

        // Extract phone (prefer phone over mobile)
        $phone = $crmContact['phone'] ?? $crmContact['mobile'] ?? null;

        // Extract company from parent_id if available (for individual contacts)
        $company = null;
        if (!$isCompany && isset($crmContact['parent_id']) && is_array($crmContact['parent_id'])) {
            $company = $crmContact['parent_id'][1] ?? null; // parent_id is [id, name]
        }

        // Build address if available
        $address = '';
        if (!empty($crmContact['street'])) {
            $address .= $crmContact['street'];
        }
        if (!empty($crmContact['city'])) {
            $address .= ($address ? ', ' : '') . $crmContact['city'];
        }
        if (!empty($crmContact['zip'])) {
            $address .= ($address ? ' ' : '') . $crmContact['zip'];
        }

        // For company contacts, add additional fields
        $additionalData = [];
        if ($isCompany) {
            $additionalData = [
                'contact_type' => 'company',
                'vat_number' => $crmContact['vat'] ?? null,
                'title' => $crmContact['title'] ?? null,
                'function' => $crmContact['function'] ?? null,
            ];
        } else {
            $additionalData = [
                'contact_type' => 'individual',
                'title' => $crmContact['title'] ?? null,
                'function' => $crmContact['function'] ?? null,
                'language' => $crmContact['lang'] ?? null,
                'timezone' => $crmContact['tz'] ?? null,
            ];
        }

        return array_merge([
            'name' => $name,
            'email' => $email,
            'phone' => $phone,
            'company' => $company,
            'address' => $address ?: null,
            'website' => $crmContact['website'] ?? null,
            'notes' => $crmContact['comment'] ?? null,
            'crm_id' => $crmContact['id'] ?? null,
            'crm_data' => $crmContact,
            'created_at' => $crmContact['create_date'] ?? now(),
            'updated_at' => $crmContact['write_date'] ?? now()
        ], $additionalData);
    }

    /**
     * Get CRM type
     */
    public function getType(): string
    {
        return 'odoo';
    }

    /**
     * Get CRM display name
     */
    public function getDisplayName(): string
    {
        return 'Odoo CRM';
    }

    /**
     * Get required credentials
     */
    public function getRequiredCredentials(): array
    {
        return [
            'url' => 'Odoo URL (e.g., https://your-company.odoo.com)',
            'database' => 'Database Name',
            'username' => 'Username',
            'password' => 'Password'
        ];
    }

    /**
     * Validate credentials
     */
    public function validateCredentials(array $credentials): bool
    {
        $required = ['url', 'database', 'username', 'password'];
        
        foreach ($required as $field) {
            if (empty($credentials[$field])) {
                return false;
            }
        }

        // Validate URL format
        if (!filter_var($credentials['url'], FILTER_VALIDATE_URL)) {
            return false;
        }

        return true;
    }
}