<?php

namespace Modules\Contacts\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Modules\Contacts\Services\CrmIntegration\CrmPluginRegistry;

class StoreCrmConnectionRequest extends FormRequest
{
    protected $crmRegistry;

    public function __construct(CrmPluginRegistry $crmRegistry)
    {
        $this->crmRegistry = $crmRegistry;
    }

    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return auth()->check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules = [
            'name' => 'required|string|max:255',
            'type' => 'required|string|in:' . implode(',', $this->crmRegistry->getAvailableTypes()),
            'is_active' => 'boolean',
            'field_mapping' => 'array',
            'field_mapping.*' => 'string|nullable',
        ];

        // Add dynamic credential validation based on CRM type
        if ($this->has('type') && $this->crmRegistry->has($this->type)) {
            $connector = $this->crmRegistry->get($this->type);
            $requiredCredentials = $connector->getRequiredCredentials();

            foreach ($requiredCredentials as $field => $config) {
                // Handle both string labels and array configs
                if (is_string($config)) {
                    // Simple string label - make it required
                    $rule = 'required|string';
                } else {
                    // Array config with validation rules
                    $rule = $config['required'] ? 'required' : 'nullable';
                    $rule .= '|string';
                    
                    if (isset($config['min_length'])) {
                        $rule .= '|min:' . $config['min_length'];
                    }
                    
                    if (isset($config['max_length'])) {
                        $rule .= '|max:' . $config['max_length'];
                    }
                }

                $rules["credentials.{$field}"] = $rule;
            }
        }

        return $rules;
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array
     */
    public function messages()
    {
        return [
            'name.required' => __('crm.validation.name_required'),
            'name.string' => __('crm.validation.name_string'),
            'name.max' => __('crm.validation.name_max'),
            'type.required' => __('crm.validation.type_required'),
            'type.in' => __('crm.validation.type_invalid'),
            'is_active.boolean' => __('crm.validation.is_active_boolean'),
            'field_mapping.array' => __('crm.validation.field_mapping_array'),
            'credentials.required' => __('crm.validation.credentials_required'),
            'credentials.string' => __('crm.validation.credentials_string'),
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $attributes = [
            'name' => __('crm.attributes.name'),
            'type' => __('crm.attributes.type'),
            'is_active' => __('crm.attributes.is_active'),
            'field_mapping' => __('crm.attributes.field_mapping'),
        ];

        // Add dynamic credential attributes
        if ($this->has('type') && $this->crmRegistry->has($this->type)) {
            $connector = $this->crmRegistry->get($this->type);
            $requiredCredentials = $connector->getRequiredCredentials();

            foreach ($requiredCredentials as $field => $config) {
                $attributes["credentials.{$field}"] = $config['label'] ?? $field;
            }
        }

        return $attributes;
    }

    /**
     * Configure the validator instance.
     *
     * @param  \Illuminate\Validation\Validator  $validator
     * @return void
     */
    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            // Validate credentials format
            if ($this->has('type') && $this->has('credentials') && $this->crmRegistry->has($this->type)) {
                $connector = $this->crmRegistry->get($this->type);
                
                if (!$connector->validateCredentials($this->credentials)) {
                    $validator->errors()->add('credentials', __('crm.validation.credentials_invalid_format'));
                }
            }
        });
    }
}