<?php

namespace Modules\Contacts\Http\Controllers;

use Modules\Contacts\Models\Contact;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Http\Controllers\Controller;
use App\Models\Plans;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Contacts\Exports\ContactsExport;
use Modules\Contacts\Imports\ContactsImport;
use Modules\Contacts\Models\Country;
use Modules\Contacts\Models\Field;
use Modules\Contacts\Models\Group;
use Modules\Contacts\Http\Requests\StoreContactRequest;
use Modules\Contacts\Http\Requests\UpdateContactRequest;
use Modules\Contacts\Services\PhoneNormalizationService;
use Modules\Contacts\Services\DemoCsvService;
use Illuminate\Support\Facades\Log;


class Main extends Controller
{
    /**
     * Provide class.
     */
    private $provider = Contact::class;

    /**
     * Web RoutePath for the name of the routes.
     */
    private $webroute_path = 'contacts.';

    /**
     * View path.
     */
    private $view_path = 'contacts::contacts.';

    /**
     * Parameter name.
     */
    private $parameter_name = 'contact';

    /**
     * Title of this crud.
     */
    private $title = 'contact';

    /**
     * Title of this crud in plural.
     */
    private $titlePlural = 'contacts';

    /**
     * Service instances
     */
    protected PhoneNormalizationService $phoneService;
    protected DemoCsvService $demoCsvService;

    /**
     * Constructor with dependency injection
     */
    public function __construct(
        PhoneNormalizationService $phoneService,
        DemoCsvService $demoCsvService
    ) {
        $this->phoneService = $phoneService;
        $this->demoCsvService = $demoCsvService;
    }

    private function hasAccessToAIBots(){
        return false;
        $allowedPluginsPerPlan = auth()->user()->company?auth()->user()->company->getPlanAttribute()['allowedPluginsPerPlan']:[];
        if($allowedPluginsPerPlan==null||in_array("flowiseai",$allowedPluginsPerPlan)){
            return true;
        }else{
            return false;
        }
    }

    private function getFields($class='col-md-4',$getCustom=true)
    {
        $fields=[];

        //Avatar
        $fields[0]=['class'=>$class, 'ftype'=>'image', 'name'=>'Avatar', 'id'=>'avatar','style'=>'width: 200px; height:200'];
        
        //Add name field
        $fields[1]=['class'=>$class, 'ftype'=>'input', 'name'=>'Name', 'id'=>'name', 'placeholder'=>'Enter name', 'required'=>true];

        //Add phone field
        $fields[2]=['class'=>$class, 'ftype'=>'input','type'=>"phone", 'name'=>'Phone', 'id'=>'phone', 'placeholder'=>'Enter phone', 'required'=>true , 'oninput'=>"this.value=this.value.replace(/[^0-9+]/g,'').replace(/(?!^)\+/g,'');"];

        //Groups
        $fields[3]=['class'=>$class, 'multiple'=>true, 'classselect'=>"select2init", 'ftype'=>'select', 'name'=>'Groups', 'id'=>'groups[]', 'placeholder'=>'Select group', 'data'=>Group::get()->pluck('name','id'), 'required'=>true];
        
        //Country
        $fields[4]=['class'=>$class, 'ftype'=>'select', 'name'=>'Country', 'id'=>'country_id', 'placeholder'=>'Select country', 'data'=>Country::get()->pluck('name','id'), 'required'=>true];

        //Email
        $fields[5]=['class'=>$class, 'ftype'=>'input', 'name'=>'Email', 'id'=>'email', 'placeholder'=>'Enter email', 'required'=>false];
        
        //Assigned Agent (Only for owners)
        $customFieldStart=5;
        if(auth()->user()->hasRole('owner')) {
            $customFieldStart=6;
            $agents = $this->getCompany()->staff()->get()->pluck('name', 'id')->toArray();
            $agents = ['' => 'Unassigned'] + $agents; // Add unassigned option
            $fields[6]=['class'=>$class, 'ftype'=>'select', 'name'=>'Assigned Agent', 'id'=>'user_id', 'placeholder'=>'Select agent', 'data'=>$agents, 'required'=>false];
        }
        
        //AI Bot enabled
        $customFieldStart=$customFieldStart;

       if($this->hasAccessToAIBots()){
            $customFieldStart++;
            $fields[$customFieldStart]=['class'=>$class, 'ftype'=>'bool', 'name'=>'Enable AI bot Replies', 'id'=>'enabled_ai_bot', 'placeholder'=>'AI Bot replies enabled', 'required'=>false];
        }


        if($getCustom){
            $customFields=Field::get()->toArray();
            $i=$customFieldStart;   
            foreach ($customFields as $filedkey => $customField) {
                $i++;
                $fields[$i]=['class'=>$class, 'ftype'=>'input', 'type'=>$customField['type'], 'name'=>__($customField['name']), 'id'=>"custom[".$customField['id']."]", 'placeholder'=>__($customField['name']), 'required'=>false];
    
            }
        }
        

        //Return fields
        return $fields;
    }


    private function getFilterFields(){
        $fields=$this->getFields('col-md-3',false);
        unset($fields[0]);
        $fields[1]['required']=false;
        $fields[2]['required']=false;

        $fields[3]['required']=false;
        $fields[3]['multiple']=false;
        $fields[3]['id']='group';
        unset($fields[3]['multiple']);

        $fields[4]['required']=false;
        $fields[4]['multiple']=false;
        unset($fields[4]['multiple']);

        $fields[5]['required']=false;

        // Agent filter field (only for owners)
        if(auth()->user()->hasRole('owner')) {
            $fields[6]['required']=false;
            $fields[6]['id']='agent_filter';
            unset($fields[6]['multiple']);
        } else {
            unset($fields[6]);
        }

        $nextIndex = auth()->user()->hasRole('owner') ? 7 : 6;
        unset($fields[$nextIndex]); // Remove AI bot field from filters

        $fields[$nextIndex]=['class'=>'col-md-3', 'ftype'=>'select', 'name'=>'Subscribed', 'id'=>'subscribed', 'placeholder'=>'Select status', 'data'=>['1'=>"Subscribed",'0'=>"Opted out"], 'required'=>false];

        return $fields;
    }

    /**
     * Auth checker functin for the crud.
     */
    private function authChecker()
    {
        $this->ownerAndStaffOnly();
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $this->authChecker();

        $items=$this->provider::orderBy('id', 'desc');
        
        // Filter contacts for agents - they only see assigned contacts
        if(auth()->user()->hasRole('staff')) {
            $items = $items->where('user_id', auth()->id());
        }
        if(isset($_GET['name'])&&strlen($_GET['name'])>1){
            $items=$items->where('name',  'like', '%'.$_GET['name'].'%');
        }
        if(isset($_GET['phone'])&&strlen($_GET['phone'])>1){
            $items=$items->where('phone',  'like', '%'.$_GET['phone'].'%');
        }

        if(isset($_GET['email'])&&strlen($_GET['email'])>1){
            $items=$items->where('email',  'like', '%'.$_GET['email'].'%');
        }

        if(isset($_GET['group'])&&strlen($_GET['group']."")>0){
            $items=$items->whereHas('groups', function ($query) {
                $query->where('groups.id',  $_GET['group']);
            });
        }
        if(isset($_GET['country_id'])&&strlen($_GET['country_id'])>0){
            $items=$items->where('country_id', $_GET['country_id'] );
        }
        
        // Filter by assigned agent (only for owners)
        if(auth()->user()->hasRole('owner') && isset($_GET['agent_filter']) && strlen($_GET['agent_filter'])>0){
            if($_GET['agent_filter'] == 'unassigned') {
                $items=$items->whereNull('user_id');
            } else {
                $items=$items->where('user_id', $_GET['agent_filter']);
            }
        }
        
        //Check subscribed
        if(isset($_GET['subscribed'])&&strlen($_GET['subscribed'])>0){
            $items=$items->where('subscribed', $_GET['subscribed'] );
        }

        if(isset($_GET['report'])){
            // Only allow owners to export contacts
            if (!auth()->user()->hasRole('owner')) {
                abort(403, 'Unauthorized action. Only owners can export contacts.');
            }
            //dd($items->with(['fields','groups'])->get());      
            return $this->exportCSV($items->with(['fields','groups'])->get());
            
        }
        $totalItems=$items->count();
        $items=$items->paginate(config('settings.paginate'));

        // Prepare setup array
        $setup = [
            'usefilter'=>true,
            'title'=>__('crud.item_managment', ['item'=>__($this->titlePlural)]),
            'subtitle'=>$totalItems==1?__('1 Contact'):$totalItems." ".__('Contacts'),
            'action_link'=>route($this->webroute_path.'create'),
            'action_name'=>__('crud.add_new_item', ['item'=>__($this->title)]),
            'action_link2'=>route($this->webroute_path.'groups.index'),
            'action_name2'=>__('Groups'),
            'action_link3'=>route($this->webroute_path.'fields.index'),
            'action_name3'=>__('Fields'),
            'items'=>$items,
            'item_names'=>$this->titlePlural,
            'webroute_path'=>$this->webroute_path,
            'fields'=>$this->getFields(),
            'filterFields'=>$this->getFilterFields(),
            'custom_table'=>true,
            'parameter_name'=>$this->parameter_name,
            'parameters'=>count($_GET) != 0,
            'groups'=>Group::get(),
        ];

        // Only show export button for owners (not for agents/staff)
        if (auth()->user()->hasRole('owner')) {
            $setup['action_link4'] = route($this->webroute_path.'index',['report'=>true]);
            $setup['action_name4'] = __('Export');
        }

        return view($this->view_path.'index', ['setup' => $setup]);
    }

    public function exportCSV($contactsToDownload){
        $items=[];
        $cf=Field::get();
        foreach ($contactsToDownload as $key => $contact) {
            $item = [
                'id'=>$contact->id,
                'name'=>$contact->name,
                'phone'=>$contact->phone,
                'avatar'=>$contact->avatar,
                'email'=>$contact->email,
            ];

            foreach( $cf as $keycf => $scf) {
                $item[$scf->name]="";
                foreach ($contact->fields as $key => $value) {
                    if($scf->name==$value['name']){
                        $item[$value['name']]=$value['pivot']['value'];
                    }
                    
                }
            }

           
            array_push($items, $item);
        }
        return Excel::download(new ContactsExport($items), 'contacts_'.time().'.csv', \Maatwebsite\Excel\Excel::CSV);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $this->authChecker();


        return view($this->view_path.'edit', ['setup' => [
            'title'=>__('crud.new_item', ['item'=>__($this->title)]),
            'action_link'=>route($this->webroute_path.'index'),
            'action_name'=>__('crud.back'),
            'iscontent'=>true,
            'action'=>route($this->webroute_path.'store'),
        ],
        'fields'=>$this->getFields() ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    // public function store(Request $request)
    public function store(StoreContactRequest $request)
    {
        $this->authChecker();
        
        //Create new contact
        $contact = $this->provider::create([
            'name' => $request->name,
            'phone' => $request->phone,
            'email' => $request->email,
            'user_id' => (!empty($request->user_id) && $request->user_id !== '') ? $request->user_id : null, // Assign agent if provided
        ]);
        $contact->save();

        if($request->has('avatar')){
            if(config('settings.use_s3_as_storage',false)){
                //S3
                $contact->avatar=Storage::disk('s3')->url($request->avatar->storePublicly("uploads/".$contact->company_id."/contacts",'s3'));
            }else{
                $contact->avatar=Storage::disk('public_media_upload')->url($request->avatar->store(null,'public_media_upload'));
            }

            $contact->update();
        }

        // Attaching groups to the contact
        $contact->groups()->attach($request->groups);

        if(isset($request->custom)){
            $this->syncCustomFieldsToContact($request->custom,$contact);
        }

        return redirect()->route($this->webroute_path.'index')->withStatus(__('crud.item_has_been_added', ['item'=>__($this->title)]));
    }

    

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Contact  $contacts
     * @return \Illuminate\Http\Response
     */
    public function edit(Contact $contact)
    {
        $this->authChecker();
        
        // Agents can only edit their assigned contacts
        if(auth()->user()->hasRole('staff') && $contact->user_id != auth()->id()) {
            abort(403, 'Unauthorized action. You can only edit contacts assigned to you.');
        }

        $fields = $this->getFields();
        $fields[0]['value'] = $contact->avatar;
        $fields[1]['value'] = $contact->name;
        $fields[2]['value'] = $contact->phone;

        $fields[3]['multipleselected'] = $contact->groups->pluck('id')->toArray();
        $fields[4]['value'] = $contact->country_id;

        $fields[5]['value'] = $contact->email;
        
        // Set assigned agent value (only for owners)
        if(auth()->user()->hasRole('owner')){
            $fields[6]['value'] = $contact->user_id;
        }

        $fieldIndex = auth()->user()->hasRole('owner') ? 7 : 6;
        if($this->hasAccessToAIBots()){
            $fields[$fieldIndex]['value'] = $contact->enabled_ai_bot.""=="1";
        }


        $customFieldsValues=$contact->fields->toArray();
        foreach ($customFieldsValues as $key => $fieldWithPivot) {
            foreach ( $fields as $key => &$formField) {
               if($formField['id']=="custom[".$fieldWithPivot['id']."]"){
                $formField['value']=$fieldWithPivot['pivot']['value'];
               }
            }
        }


        $parameter = [];
        $parameter[$this->parameter_name] = $contact->id;
        $title=__('crud.edit_item_name', ['item'=>__($this->title), 'name'=>$contact->name]);
        return view($this->view_path.'edit', ['setup' => [
            'title'=>$title ." - ". ($contact->subscribed=="1" ? __('Subscribed') : __('Opted out')),
            'action_link'=>route($this->webroute_path.'index'),
            'action_name'=>__('crud.back'),
            'iscontent'=>true,
            'isupdate'=>true,
            'action'=>route($this->webroute_path.'update', $parameter),
        ],
        'fields'=>$fields, ]);
    }
        
        
        
    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Contact  $contacts
     * @return \Illuminate\Http\Response
     */
    // public function update(Request $request, $id)
    public function update(UpdateContactRequest $request, $id)
    {
        $this->authChecker();
        
        $item = $this->provider::findOrFail($id);
        
        // Agents can only update their assigned contacts
        if(auth()->user()->hasRole('staff') && $item->user_id != auth()->id()) {
            abort(403, 'Unauthorized action. You can only update contacts assigned to you.');
        }
        
        $item->name = $request->name;
        $item->phone = $request->phone;
        $item->country_id = $request->country_id;
        $item->email = $request->email;
        
        // Only owners can change agent assignment
        if(auth()->user()->hasRole('owner')){
            $item->user_id = (!empty($request->user_id) && $request->user_id !== '') ? $request->user_id : null;
        }
        
        if($this->hasAccessToAIBots()){
            $item->enabled_ai_bot = $request->enabled_ai_bot=="true";
        }


        if($request->has('avatar')){
            if(config('settings.use_s3_as_storage',false)){
                //S3
                $item->avatar=Storage::disk('s3')->url($request->avatar->storePublicly("uploads/".$item->company_id."/contacts",'s3'));
            }else{
                $item->avatar=Storage::disk('public_media_upload')->url($request->avatar->store(null,'public_media_upload'));
            }

            
        }

        $item->update();

        if(isset($request->custom)){
            $this->syncCustomFieldsToContact($request->custom,$item);
        }
        

        // Attaching groups to the contact
        $item->groups()->sync($request->groups);
        $item->update();

        return redirect()->route($this->webroute_path.'index')->withStatus(__('crud.item_has_been_updated', ['item'=>__($this->title)]));
    }


    public function syncCustomFieldsToContact($fields,$contact){
        $contact->fields()->sync([]);
        foreach ($fields as $key => $value) {
            if($value){
                $contact->fields()->attach($key, ['value' => $value]);
            }
          
        }
        $contact->update();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Contact  $contacts
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $this->authChecker();
        $item = $this->provider::findOrFail($id);
        
        // Agents can only delete their assigned contacts
        if(auth()->user()->hasRole('staff') && $item->user_id != auth()->id()) {
            abort(403, 'Unauthorized action. You can only delete contacts assigned to you.');
        }
        
        $item->delete();
        return redirect()->route($this->webroute_path.'index')->withStatus(__('crud.item_has_been_removed', ['item'=>__($this->title)]));
    }

    public function bulkremove($ids)
    {
        $this->authChecker();
        $ids = explode(",", $ids);
        $this->provider::destroy($ids);

        // Return a JSON response
        return response()->json([
            'status' => 'success',
            'message' => __('crud.items_have_been_removed', ['item' => __($this->titlePlural)])
        ], 200);
    }

    public function subscribe($ids)
    {
        $this->authChecker();
        $ids = explode(",", $ids);
        $this->provider::whereIn('id', $ids)->update(['subscribed' => 1]);

        // Return a JSON response
        return response()->json([
            'status' => 'success',
            'message' => __('crud.item_has_been_updated', ['item' => __($this->titlePlural)])
        ], 200);
    }

    public function unsubscribe($ids)
    {
        $this->authChecker();
        $ids = explode(",", $ids);
        $this->provider::whereIn('id', $ids)->update(['subscribed' => 0]);

        // Return a JSON response
        return response()->json([
            'status' => 'success',
            'message' => __('crud.item_has_been_updated', ['item' => __($this->titlePlural)])
        ], 200);
    }

    

    public function assigntogroup($ids)
    {
        $this->authChecker();
        $ids = explode(",", $ids);
        $group = Group::find($_GET['group_id']);

        if (!$group) {
            // Group not found, return an error response
            return response()->json([
                'status' => 'error',
                'message' => __('No group selected')
            ], 404);
        }

        $group->contacts()->syncWithoutDetaching($ids);

        // Return a JSON response
        return response()->json([
            'status' => 'success',
            'message' => __('crud.items_has_been_updated', ['item' => __($this->titlePlural)])
        ], 200);
    }

    public function removefromgroup($ids)
    {
        $this->authChecker();
        $ids = explode(",", $ids);
        $group = Group::find($_GET['group_id']);

        if (!$group) {
            // Group not found, return an error response
            return response()->json([
                'status' => 'error',
                'message' => __('No group selected')
            ], 404);
        }

        $group->contacts()->detach($ids);

        // Return a JSON response
        return response()->json([
            'status' => 'success',
            'message' => __('crud.items_has_been_updated', ['item' => __($this->titlePlural)])
        ], 200);
    }

    /**
     * Assign contacts to an agent (bulk action)
     */
    public function assigntoagent($ids)
    {
        // Only owners can assign contacts to agents
        if (!auth()->user()->hasRole('owner')) {
            abort(403, 'Unauthorized action. Only owners can assign contacts to agents.');
        }

        $ids = explode(",", $ids);
        $agentId = isset($_GET['agent_id']) && $_GET['agent_id'] !== '' ? $_GET['agent_id'] : null;

        // Validate agent belongs to the company (only if assigning, not unassigning)
        if ($agentId !== null) {
            $agent = User::find($agentId);
            if (!$agent || $agent->company_id != session('company_id')) {
                return response()->json([
                    'status' => 'error',
                    'message' => __('Invalid agent selected')
                ], 404);
            }
        }

        // Update contacts with agent assignment (null for unassign)
        $this->provider::whereIn('id', $ids)
            ->where('company_id', session('company_id'))
            ->update(['user_id' => $agentId]);

        $message = $agentId 
            ? __('Contacts assigned to agent successfully')
            : __('Contacts unassigned from agents successfully');

        return response()->json([
            'status' => 'success',
            'message' => $message
        ], 200);
    }

    public function importindex(){
        $groups=Group::pluck('name','id');
        $crmConnections = \Modules\Contacts\Models\CrmConnection::active()->get();
        return view("contacts::".$this->webroute_path.'import',['groups'=>$groups, 'crmConnections'=>$crmConnections]);
    }

    public function import(Request $request){
        $this->authChecker();
        
        try {
            $import = new ContactsImport();
            $import->resetStats();
            
            Excel::import($import, $request->csv);
            
            $stats = $import->getImportStats();
            
            // Assign to group if specified
            if($request->group){
                $csvData = Excel::toArray(new ContactsImport, $request->csv);
                $phoneNumbers = array_column($csvData[0], 'phone');
                
                // Normalize phone numbers for group assignment
                $normalizedPhones = array_map(function($phone) {
                    return $this->phoneService->normalizePhone($phone);
                }, $phoneNumbers);
                
                $contactToApply = $this->provider::whereIn('phone', $normalizedPhones)
                    ->where('company_id', session('company_id'))
                    ->pluck('id');
                
                if($contactToApply->isNotEmpty()){
                    $group = Group::find($request->group);
                    if($group) {
                        $group->contacts()->detach($contactToApply);
                        $group->contacts()->attach($contactToApply);
                    }
                }
            }
            
            $message = "Import completed: {$stats['imported']} imported, {$stats['updated']} updated, {$stats['skipped']} skipped";
            if (!empty($stats['errors'])) {
                $message .= ". " . count($stats['errors']) . " errors occurred.";
            }
            
            return redirect()->route($this->webroute_path.'index')->withStatus(__($message));
            
        } catch (\Exception $e) {
            Log::error('Contact import failed', [
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
                'company_id' => session('company_id')
            ]);
            
            return redirect()->route($this->webroute_path.'import.index')
                ->withError(__('Import failed: ' . $e->getMessage()));
        }
    }

    /**
     * Show CRM import interface.
     */
    public function crmImportIndex(Request $request)
    {
        $this->authChecker();
        
        $crmConnections = \Modules\Contacts\Models\CrmConnection::active()->get();
        $groups = Group::pluck('name', 'id');
        
        // Handle pre-selected connection from URL parameter
        $selectedConnection = null;
        if ($request->has('connection')) {
            $selectedConnection = \Modules\Contacts\Models\CrmConnection::find($request->connection);
        }
        
        return view('contacts::contacts.crm-import', compact('crmConnections', 'groups', 'selectedConnection'));
    }

    /**
     * Start CRM import process.
     */
    public function crmImport(\Modules\Contacts\Http\Requests\CrmImportRequest $request)
    {
        $this->authChecker();

        try {
            $crmImportService = app(\Modules\Contacts\Services\EnhancedCrmImportService::class);
            
            $syncHistory = $crmImportService->importContacts(
                $request->crm_connection_id,
                $request->validated()
            );

            return response()->json([
                'success' => true,
                'message' => __('crm.import_started_successfully'),
                'sync_history_id' => $syncHistory->id,
            ]);

        } catch (\Exception $e) {
            \Log::error('CRM import failed', [
                'error' => $e->getMessage(),
                'connection_id' => $request->crm_connection_id,
            ]);

            // Provide more specific error messages
            $errorMessage = $e->getMessage();
            if (strpos($errorMessage, 'Invalid CRM credentials') !== false) {
                $message = __('crm.invalid_credentials_error');
            } elseif (strpos($errorMessage, 'Failed to connect to CRM') !== false) {
                $message = __('crm.connection_error');
            } else {
                $message = __('crm.import_failed');
            }

            return response()->json([
                'success' => false,
                'message' => $message,
                'error_details' => $errorMessage,
            ], 500);
        }
    }

    /**
     * Get CRM import progress.
     */
    public function crmImportProgress(Request $request)
    {
        $this->authChecker();

        $request->validate([
            'sync_history_id' => 'required|integer|exists:crm_sync_history,id',
        ]);

        try {
            $crmImportService = app(\Modules\Contacts\Services\EnhancedCrmImportService::class);
            
            $progress = $crmImportService->getImportProgress($request->sync_history_id);

            return response()->json([
                'success' => true,
                'progress' => $progress,
            ]);

        } catch (\Exception $e) {
            \Log::error('Failed to get CRM import progress', [
                'error' => $e->getMessage(),
                'sync_history_id' => $request->sync_history_id,
            ]);

            return response()->json([
                'success' => false,
                'message' => __('crm.progress_fetch_failed'),
            ], 500);
        }
    }

    /**
     * Cancel CRM import.
     */
    public function crmImportCancel(Request $request)
    {
        $this->authChecker();

        $request->validate([
            'sync_history_id' => 'required|integer|exists:crm_sync_history,id',
        ]);

        try {
            $crmImportService = app(\Modules\Contacts\Services\EnhancedCrmImportService::class);
            
            $cancelled = $crmImportService->cancelImport($request->sync_history_id);

            if ($cancelled) {
                return response()->json([
                    'success' => true,
                    'message' => __('crm.import_cancelled_successfully'),
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => __('crm.import_cannot_be_cancelled'),
                ], 400);
            }

        } catch (\Exception $e) {
            \Log::error('Failed to cancel CRM import', [
                'error' => $e->getMessage(),
                'sync_history_id' => $request->sync_history_id,
            ]);

            return response()->json([
                'success' => false,
                'message' => __('crm.import_cancellation_failed'),
            ], 500);
        }
    }

    /**
     * Show import history.
     */
    public function importHistory()
    {
        $this->authChecker();
        
        $syncHistory = \Modules\Contacts\Models\CrmSyncHistory::with(['crmConnection'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('contacts::contacts.import-history', compact('syncHistory'));
    }

    /**
     * Show CRM import history.
     */
    public function crmImportHistory()
    {
        $this->authChecker();
        
        $syncHistory = \Modules\Contacts\Models\CrmSyncHistory::with(['crmConnection'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('contacts::contacts.import-history', compact('syncHistory'));
    }

    public function crmImportHistoryDetails(\Modules\Contacts\Models\CrmSyncHistory $syncHistory)
    {
        $this->authChecker();
        
        return view('contacts::contacts.import-history-details', compact('syncHistory'));
    }

    /**
     * Download demo CSV template for contact import
     */
    public function downloadDemoCsv(Request $request)
    {
        try {
            // No authentication required for demo CSV download
            
            // Log download activity for analytics (if user is logged in)
            if (auth()->check()) {
                Log::info('Demo CSV downloaded', [
                    'user_id' => auth()->id(),
                    'company_id' => session('company_id'),
                    'timestamp' => now(),
                    'user_agent' => $request->userAgent(),
                    'ip' => $request->ip()
                ]);
            } else {
                Log::info('Demo CSV downloaded (anonymous)', [
                    'timestamp' => now(),
                    'user_agent' => $request->userAgent(),
                    'ip' => $request->ip()
                ]);
            }
            
            return $this->demoCsvService->downloadDemoCsv();
            
        } catch (\Exception $e) {
            Log::error('Demo CSV download failed', [
                'user_id' => auth()->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to generate demo CSV. Please try again.'
            ], 500);
        }
    }

    /**
     * Get CSV template information for the import page
     */
    public function getCsvTemplateInfo()
    {
        try {
            $this->authChecker();
            
            $templateInfo = $this->demoCsvService->getCompanySpecificTemplate();
            
            return response()->json([
                'status' => 'success',
                'data' => $templateInfo
            ]);
            
        } catch (\Exception $e) {
            Log::error('Failed to get CSV template info', [
                'user_id' => auth()->id(),
                'error' => $e->getMessage()
            ]);
            
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get template information.'
            ], 500);
        }
    }
}
