<script>
    "use strict";
    var templateManager = null;

    window.onload = function () {
        Vue.config.devtools = true;

        templateManager = new Vue({
            el: '#template_creator',
            data: {
                touched: {},
                template_name: "",
                category: "MARKETING",
                language: "",
                headerType: "none",
                headerText: "",
                headerImage: "",
                headerVideo: "",
                headerPdf: "",
                headerHandle: "",
                headervariableAdded: false,
                headerExampleVariable: "",
                bodyText: "",
                bodyRatioError: false,
                bodyExampleVariable: [],
                footerText: "",
                quickReplies: [],
                vistiWebsite: [],
                dynamicUrls: [],
                hasPhone: false,
                dialCode: "",
                phoneNumber: "",
                callPhoneButtonText: "",
                copyOfferCode: false,
                offerCode: "",
                isSending: false,
                validation: {
                    name: false,
                    category: true, // Has default value MARKETING
                    language: false,
                    headerText: true, // Optional
                    body: false,
                    footer: true, // Optional
                    quickReplies: true,
                    websiteButtons: true,
                    dynamicUrls: true,
                    phone: true
                },
            },
            watch: {
                template_name: function (newVal, oldVal) {
                    //Clean name
                    this.template_name = newVal.replace(/\s/g, '_').replace(/-/g, '_').toLowerCase();
                    // Update validation
                    this.validation.name = newVal.length > 0;
                },
                category: function (newVal, oldVal) {
                    this.validation.category = newVal && newVal.length > 0;
                },
                language: function (newVal, oldVal) {
                    this.validation.language = newVal && newVal.length > 0;
                },
                bodyText: function (val) {
                    this.checkVariableRatio();
                    // Update validation - body is required and must be <= 1024
                    this.validation.body = val.length > 0 && val.length <= 1024 && !this.bodyRatioError;
                },
                headerText: function (val) {
                    this.headervariableAdded = (val.indexOf('@{{1}}') > -1);
                    // Update validation - header is optional but if used, must be <= 60
                    this.validation.headerText = val.length <= 60;
                },
                footerText: function (val) {
                    // Footer is optional but if used, must be <= 60
                    this.validation.footer = val.length <= 60;
                },
                quickReplies: {
                    handler: function (val) {
                        // All quick replies must be <= 20 chars (Meta limit)
                        this.validation.quickReplies = val.every(function (reply) {
                            return reply.length <= 20;
                        });
                    },
                    deep: true
                },
                vistiWebsite: {
                    handler: function (val) {
                        var self = this;
                        // All website buttons must have valid title (<=25) and valid URL
                        this.validation.websiteButtons = val.every(function (btn) {
                            return btn.title.length <= 25 && self.isValidURL(btn.url);
                        });
                    },
                    deep: true
                },
                dynamicUrls: {
                    handler: function (val) {
                        var self = this;
                        // All dynamic URLs must have valid text (<=25) and valid URL
                        this.validation.dynamicUrls = val.every(function (btn) {
                            return btn.text.length <= 25 && self.isValidURL(btn.url);
                        });
                    },
                    deep: true
                }
            },
            methods: {
                // --- Validation Methods ---
                isValidURL: function (str) {
                    var pattern = new RegExp('^(https?:\\/\\/)' + // protocol (REQUIRED)
                        '((([a-z\\d]([a-z\\d-]*[a-z\\d])*)\\.)+[a-z]{2,}|' + // domain name
                        '((\\d{1,3}\\.){3}\\d{1,3}))' + // OR ip (v4) address
                        '(\\:\\d+)?(\\/[-a-z\\d%_.~+]*)*' + // port and path
                        '(\\?[;&a-z\\d%_.~+=-]*)?' + // query string
                        '(\\#[-a-z\\d_]*)?$', 'i'); // fragment locator
                    return !!pattern.test(str);
                },
                // Validate phone number for Meta API (must be digits only, 7-15 digits)
                isValidPhoneNumber: function () {
                    if (!this.hasPhone) return true;
                    // Clean the dial code (remove + if present, should just be digits)
                    var cleanDialCode = this.dialCode.replace(/[^0-9]/g, '');
                    var cleanNumber = this.phoneNumber.replace(/[^0-9]/g, '');
                    var fullNumber = cleanDialCode + cleanNumber;
                    // Meta requires 7-15 digits
                    return fullNumber.length >= 7 && fullNumber.length <= 15;
                },
                // Get formatted phone number for Meta API (must include + prefix)
                getFormattedPhoneNumber: function () {
                    var cleanDialCode = this.dialCode.replace(/[^0-9]/g, '');
                    var cleanNumber = this.phoneNumber.replace(/[^0-9]/g, '');
                    // Meta API requires + prefix in full international format
                    return '+' + cleanDialCode + cleanNumber;
                },
                checkVariableRatio: function () {
                    let vars = this.bodyText.match(/@{{[0-9]+}}/g);
                    let varCount = vars ? vars.length : 0;

                    if (varCount === 0) {
                        this.bodyRatioError = false;
                        return;
                    }

                    // Remove variables to see pure text length
                    let pureText = this.bodyText.replace(/@{{[0-9]+}}/g, "");
                    // Remove whitespace
                    pureText = pureText.replace(/\s/g, "");

                    // The Ratio Check: Meta blocks if params are too dense.
                    if (pureText.length < (varCount * 15)) {
                        this.bodyRatioError = true;
                    } else {
                        this.bodyRatioError = false;
                    }
                },

                // --- Editor Helpers ---
                addBold: function () {
                    this.insertAtCursor('*bold*');
                },
                addItalic: function () {
                    this.insertAtCursor('_italic_');
                },
                addCode: function () {
                    this.insertAtCursor('`code`');
                },
                addVariable: function () {
                    var matches = this.bodyText.match(/@{{[0-9]+}}/g);
                    var nextVar = matches ? matches.length + 1 : 1;
                    this.insertAtCursor('@{{' + nextVar + '}}');
                },
                insertAtCursor: function (myValue) {
                    var myField = document.querySelector("#body");
                    if (!myField) return;

                    if (document.selection) {
                        myField.focus();
                        sel = document.selection.createRange();
                        sel.text = myValue;
                    }
                    else if (myField.selectionStart || myField.selectionStart == '0') {
                        var startPos = myField.selectionStart;
                        var endPos = myField.selectionEnd;
                        this.bodyText = this.bodyText.substring(0, startPos)
                            + myValue
                            + this.bodyText.substring(endPos, this.bodyText.length);
                    } else {
                        this.bodyText += myValue;
                    }
                },
                addHeaderVariable: function () {
                    if (this.headerText.indexOf('@{{1}}') == -1) {
                        this.headerText += ' @{{1}}';
                    }
                },

                // --- Upload Handlers ---
                handleImageUpload: function (event) {
                    const formData = new FormData();
                    formData.append('imageupload', event.target.files[0]);
                    axios.post('/templates/upload-image', formData, {
                        headers: { 'Content-Type': 'multipart/form-data' }
                    }).then(response => {
                        if (response.data.status == 'success') {
                            this.headerImage = response.data.url;
                            this.headerHandle = response.data.handle;
                        } else {
                            alert(response.data.response);
                        }
                    })
                },
                handleVideoUpload: function (event) {
                    const formData = new FormData();
                    formData.append('videoupload', event.target.files[0]);
                    axios.post('/templates/upload-video', formData, {
                        headers: { 'Content-Type': 'multipart/form-data' }
                    }).then(response => {
                        this.headerVideo = response.data.url;
                        this.headerHandle = response.data.handle;
                    })
                },
                handlePdfUpload: function (event) {
                    const formData = new FormData();
                    formData.append('pdfupload', event.target.files[0]);
                    axios.post('/templates/upload-pdf', formData, {
                        headers: { 'Content-Type': 'multipart/form-data' }
                    }).then(response => {
                        this.headerPdf = response.data.url;
                        this.headerHandle = response.data.handle;
                    })
                },


                // --- Button Logic ---
                addQuickReply: function () {
                    this.quickReplies.push("");
                },
                deleteQuickReply: function (index) {
                    this.quickReplies.splice(index, 1);
                },
                addVisitWebsite: function () {
                    if (this.vistiWebsite.length >= 2) return;
                    this.vistiWebsite.push({ title: "", url: "" });
                },
                deleteVisitWebsite: function (index) {
                    this.vistiWebsite.splice(index, 1);
                },
                addCallPhone: function () {
                    this.hasPhone = true;
                },
                deletePhone: function () {
                    this.hasPhone = false;
                    this.dialCode = "";
                    this.phoneNumber = "";
                    this.callPhoneButtonText = "";
                },
                addCopyOfferCode: function () {
                    this.copyOfferCode = true;
                },
                deleteCopyOfferCode: function () {
                    this.copyOfferCode = false;
                    this.offerCode = "";
                },
                addDynamicUrl: function () {
                    if (this.dynamicUrls.length >= 2) return;
                    this.dynamicUrls.push({ type: "dynamic", text: "", url: "", example: "" });
                },
                deleteDynamicUrl: function (index) {
                    this.dynamicUrls.splice(index, 1);
                },

                // --- Submission ---
                submitTemplate: function () {
                    // 0. Basic Template Validation (already handled by watchers)
                    // Ensure all validation flags are true before proceeding
                    if (!Object.values(this.validation).every(v => v)) {
                        alert("Please fix the highlighted errors before submitting.");
                        return;
                    }

                    // 1. Header Validation
                    if (this.headerType == 'text' && this.headerText.length > 60) {
                        alert("Header text is too long (Max 60 chars).");
                        return;
                    }

                    // 2. Body Validation
                    if (this.bodyText.length > 1024) {
                        alert("Body text is too long (Max 1024 chars).");
                        return;
                    }

                    // 3. Variable Ratio Validation
                    if (this.bodyRatioError) {
                        alert("Your message has too many variables compared to the text. Please add more text to explain your variables.");
                        return;
                    }

                    // 4. Footer Validation
                    if (this.footerText.length > 60) {
                        alert("Footer text is too long (Max 60 chars).");
                        return;
                    }

                    // 5. Button Validations
                    for (var i = 0; i < this.quickReplies.length; i++) {
                        if (this.quickReplies[i].length > 25) {
                            alert("Quick reply button text is too long (Max 25 chars).");
                            return;
                        }
                    }

                    for (var i = 0; i < this.vistiWebsite.length; i++) {
                        if (this.vistiWebsite[i]['title'].length > 25) {
                            alert("Website button text is too long (Max 25 chars).");
                            return;
                        }
                        if (!this.isValidURL(this.vistiWebsite[i]['url'])) {
                            alert("Invalid Website URL: " + this.vistiWebsite[i]['url'] + "\nMust start with http:// or https://");
                            return;
                        }
                    }

                    for (var i = 0; i < this.dynamicUrls.length; i++) {
                        if (this.dynamicUrls[i]['text'].length > 25) {
                            alert("Dynamic URL button text is too long (Max 25 chars).");
                            return;
                        }
                        if (!this.isValidURL(this.dynamicUrls[i]['url'])) {
                            alert("Invalid Dynamic URL: " + this.dynamicUrls[i]['url'] + "\nMust start with http:// or https://");
                            return;
                        }
                    }

                    if (this.hasPhone && this.callPhoneButtonText.length > 25) {
                        alert("Phone button text is too long (Max 25 chars).");
                        return;
                    }

                    // 6. Phone Number Format Validation
                    if (this.hasPhone && !this.isValidPhoneNumber()) {
                        alert("Invalid phone number format.\n\nPlease enter:\n• Country code (e.g. 971 for UAE, 1 for USA)\n• Phone number (digits only)\n\nTotal must be 7-15 digits.");
                        return;
                    }
                    // --- END VALIDATION --- //


                    if (!this.isSending) {
                        this.isSending = true;
                        var metaTemplate = {
                            "name": this.template_name,
                            "category": this.category,
                            "language": this.language,
                            "allow_category_change": true,
                        };

                        var components = [];

                        // 1. HEADER
                        if (this.headerType != 'none') {
                            var headerComponent = {
                                "type": "HEADER",
                                "format": this.headerType.toUpperCase()
                            };

                            if (this.headerType == 'text') {
                                headerComponent.text = this.headerText;
                                // Add example if variable exists
                                if (this.headerText.indexOf('@{{1}}') > -1) {
                                    headerComponent.example = {
                                        "header_text": [this.headerExampleVariable]
                                    };
                                }
                            } else {
                                // Media handles
                                headerComponent.example = { "header_handle": [this.headerHandle] };
                            }
                            components.push(headerComponent);
                        }

                        // 2. BODY
                        var bodyComponent = {
                            "type": "BODY",
                            "text": this.bodyText
                        };
                        // Body Examples
                        if (this.bodyVariables && this.bodyVariables.length > 0) {
                            bodyComponent.example = {
                                "body_text": [this.bodyExampleVariable]
                            };
                        }
                        components.push(bodyComponent);

                        // 3. FOOTER
                        if (this.footerText.length > 0) {
                            components.push({
                                "type": "FOOTER",
                                "text": this.footerText
                            });
                        }

                        // 4. BUTTONS
                        var buttonsComponent = { "type": "BUTTONS", "buttons": [] };

                        this.quickReplies.forEach(function (reply) {
                            buttonsComponent.buttons.push({ "type": "QUICK_REPLY", "text": reply });
                        });

                        this.vistiWebsite.forEach(function (website) {
                            buttonsComponent.buttons.push({ "type": "URL", "text": website.title, "url": website.url });
                        });

                        if (this.hasPhone) {
                            buttonsComponent.buttons.push({
                                "type": "PHONE_NUMBER",
                                "text": this.callPhoneButtonText,
                                "phone_number": this.getFormattedPhoneNumber()
                            });
                        }

                        if (this.copyOfferCode) {
                            buttonsComponent.buttons.push({
                                "type": "COPY_CODE",
                                "example": this.offerCode || "SAVE20"
                            });
                        }

                        this.dynamicUrls.forEach(function (dynamicUrl) {
                            buttonsComponent.buttons.push({
                                "type": "URL",
                                "text": dynamicUrl.text,
                                "url": dynamicUrl.url + "{{1}}",
                                "example": [dynamicUrl.url + dynamicUrl.example]
                            });
                        });

                        if (buttonsComponent.buttons.length > 0) {
                            components.push(buttonsComponent);
                        }

                        metaTemplate.components = components;

                        var self = this;
                        axios.post('/templates/submit', metaTemplate)
                            .then(function (response) {
                                if (response.data.status == "success") {
                                    window.location.href = '/templates?ok=1&refresh=yes';
                                } else {
                                    alert('Error: ' + JSON.stringify(response.data));
                                    self.isSending = false;
                                }
                            })
                            .catch(function (error) {
                                alert('Error: ' + error);
                                self.isSending = false;
                            });
                    }
                }
            },
            computed: {
                // Computed property to extract variables for the UI usage (like generating example input fields)
                bodyVariables: function () {
                    var match = this.bodyText.match(/@{{[0-9]+}}/g);
                    return match ? [...new Set(match)] : [];
                },
                // Computed property to enable/disable submit button
                canSubmit: function () {
                    return Object.values(this.validation).every(v => v);
                },
                // Replace body variables with example content for preview
                bodyReplacedWithExample: function () {
                    var text = this.bodyText;
                    var self = this;
                    if (this.bodyVariables && this.bodyVariables.length > 0) {
                        this.bodyVariables.forEach(function (v, index) {
                            var example = self.bodyExampleVariable[index] || '[Sample ' + (index + 1) + ']';
                            text = text.replace(new RegExp(v.replace(/[{}]/g, '\\$&'), 'g'), example);
                        });
                    }
                    return text;
                },
                // Replace header variable with example content for preview
                headerReplacedWithExample: function () {
                    var text = this.headerText;
                    if (this.headervariableAdded && this.headerExampleVariable) {
                        text = text.replace(/@{{1}}/g, this.headerExampleVariable);
                    } else if (this.headervariableAdded) {
                        text = text.replace(/@{{1}}/g, '[Sample]');
                    }
                    return text;
                }
            }
        });
    };
</script>