<?php

namespace Modules\Flowmaker\Tests\Unit;

use Tests\TestCase;
use Modules\Flowmaker\Services\SMTPValidator;
use Modules\Flowmaker\Services\ValidationResult;
use Illuminate\Support\Facades\Log;

class SMTPValidatorTest extends TestCase
{
    private SMTPValidator $validator;

    protected function setUp(): void
    {
        parent::setUp();
        $this->validator = new SMTPValidator();
    }

    /** @test */
    public function it_validates_required_fields()
    {
        // Test missing host
        $config = [
            'port' => 587,
            'username' => 'test@example.com',
            'password' => 'password'
        ];

        $result = $this->validator->validateCredentials($config);
        
        $this->assertFalse($result->isValid);
        $this->assertStringContains('Missing required SMTP field: host', $result->errorMessage);
    }

    /** @test */
    public function it_validates_port_is_numeric()
    {
        $config = [
            'host' => 'smtp.gmail.com',
            'port' => 'invalid',
            'username' => 'test@example.com',
            'password' => 'password'
        ];

        $result = $this->validator->validateCredentials($config);
        
        $this->assertFalse($result->isValid);
        $this->assertStringContains('SMTP port must be numeric', $result->errorMessage);
    }

    /** @test */
    public function it_validates_encryption_type()
    {
        $config = [
            'host' => 'smtp.gmail.com',
            'port' => 587,
            'username' => 'test@example.com',
            'password' => 'password',
            'encryption' => 'invalid'
        ];

        $result = $this->validator->validateCredentials($config);
        
        $this->assertFalse($result->isValid);
        $this->assertStringContains('Invalid encryption type', $result->errorMessage);
    }

    /** @test */
    public function it_tests_basic_connection()
    {
        // Test with a known unreachable host
        $config = [
            'host' => 'nonexistent.example.com',
            'port' => 587,
            'timeout' => 1
        ];

        $result = $this->validator->testConnection($config);
        $this->assertFalse($result);
    }

    /** @test */
    public function it_handles_empty_host_in_connection_test()
    {
        $config = [
            'host' => '',
            'port' => 587
        ];

        $result = $this->validator->testConnection($config);
        $this->assertFalse($result);
    }

    /** @test */
    public function it_sanitizes_config_for_logging()
    {
        $config = [
            'host' => 'smtp.gmail.com',
            'port' => 587,
            'username' => 'test@example.com',
            'password' => 'secret123'
        ];

        // This will trigger validation and set lastTestedConfig
        $this->validator->validateCredentials($config);
        $sanitized = $this->validator->getLastTestedConfig();
        
        $this->assertEquals('***', $sanitized['password']);
        $this->assertEquals('test@example.com', $sanitized['username']);
    }

    /** @test */
    public function it_returns_validation_result_with_timing()
    {
        $config = [
            'host' => 'smtp.gmail.com',
            'port' => 587,
            'username' => 'test@example.com',
            'password' => 'password'
        ];

        $result = $this->validator->validateCredentials($config);
        
        $this->assertInstanceOf(ValidationResult::class, $result);
        $this->assertGreaterThan(0, $result->responseTime);
        $this->assertIsArray($result->testedConfig);
    }

    /** @test */
    public function it_stores_last_error_message()
    {
        $config = [
            'host' => 'smtp.gmail.com',
            'port' => 'invalid',
            'username' => 'test@example.com',
            'password' => 'password'
        ];

        $this->validator->validateCredentials($config);
        $lastError = $this->validator->getLastError();
        
        $this->assertNotEmpty($lastError);
        $this->assertStringContains('SMTP port must be numeric', $lastError);
    }

    /** @test */
    public function validation_result_can_be_converted_to_array()
    {
        $result = new ValidationResult(
            true,
            '',
            ['host' => 'smtp.gmail.com'],
            0.5,
            'OK'
        );

        $array = $result->toArray();
        
        $this->assertIsArray($array);
        $this->assertTrue($array['isValid']);
        $this->assertEquals(0.5, $array['responseTime']);
        $this->assertEquals('OK', $array['serverResponse']);
    }

    /** @test */
    public function it_handles_missing_optional_fields_gracefully()
    {
        $config = [
            'host' => 'smtp.gmail.com',
            'port' => 587,
            'username' => 'test@example.com',
            'password' => 'password'
            // Missing encryption, timeout - should use defaults
        ];

        $result = $this->validator->validateCredentials($config);
        
        // Should not fail due to missing optional fields
        $this->assertNotNull($result);
        $this->assertInstanceOf(ValidationResult::class, $result);
    }
}