<?php

namespace Modules\Flowmaker\Tests\Feature;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Log;

class SMTPTestControllerTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create a test user for authentication
        $this->user = \App\Models\User::factory()->create();
    }

    /** @test */
    public function it_validates_required_fields_for_credential_testing()
    {
        $response = $this->actingAs($this->user, 'api')
            ->postJson('/api/flowmaker/smtp/test-credentials', []);

        $response->assertStatus(422)
            ->assertJson([
                'success' => false,
                'error' => 'Invalid SMTP configuration'
            ])
            ->assertJsonStructure([
                'success',
                'error',
                'details' => [
                    'validation_errors'
                ]
            ]);
    }

    /** @test */
    public function it_validates_port_is_numeric()
    {
        $response = $this->actingAs($this->user, 'api')
            ->postJson('/api/flowmaker/smtp/test-credentials', [
                'host' => 'smtp.gmail.com',
                'port' => 'invalid',
                'username' => 'test@gmail.com',
                'password' => 'password'
            ]);

        $response->assertStatus(422)
            ->assertJsonPath('details.validation_errors.port.0', 'The port field must be an integer.');
    }

    /** @test */
    public function it_validates_encryption_type()
    {
        $response = $this->actingAs($this->user, 'api')
            ->postJson('/api/flowmaker/smtp/test-credentials', [
                'host' => 'smtp.gmail.com',
                'port' => 587,
                'username' => 'test@gmail.com',
                'password' => 'password',
                'encryption' => 'invalid'
            ]);

        $response->assertStatus(422)
            ->assertJsonPath('details.validation_errors.encryption.0', 'The selected encryption is invalid.');
    }

    /** @test */
    public function it_tests_smtp_credentials_with_invalid_config()
    {
        // This will fail because it's not a real SMTP server
        $response = $this->actingAs($this->user, 'api')
            ->postJson('/api/flowmaker/smtp/test-credentials', [
                'host' => 'nonexistent.example.com',
                'port' => 587,
                'username' => 'test@example.com',
                'password' => 'wrongpassword',
                'encryption' => 'tls'
            ]);

        $response->assertStatus(400)
            ->assertJson([
                'success' => false
            ])
            ->assertJsonStructure([
                'success',
                'error',
                'details' => [
                    'host',
                    'port',
                    'username',
                    'encryption',
                    'response_time',
                    'tested_at',
                    'suggestions'
                ]
            ]);
    }

    /** @test */
    public function it_tests_basic_connection()
    {
        $response = $this->actingAs($this->user, 'api')
            ->postJson('/api/flowmaker/smtp/test-connection', [
                'host' => 'nonexistent.example.com',
                'port' => 587,
                'timeout' => 5
            ]);

        $response->assertStatus(400)
            ->assertJson([
                'success' => false,
                'error' => 'Cannot connect to SMTP server'
            ])
            ->assertJsonStructure([
                'success',
                'error',
                'details' => [
                    'host',
                    'port',
                    'suggestions'
                ]
            ]);
    }

    /** @test */
    public function it_validates_connection_parameters()
    {
        $response = $this->actingAs($this->user, 'api')
            ->postJson('/api/flowmaker/smtp/test-connection', [
                'port' => 587
                // Missing host
            ]);

        $response->assertStatus(422)
            ->assertJson([
                'success' => false,
                'error' => 'Invalid connection parameters'
            ]);
    }

    /** @test */
    public function it_returns_provider_suggestions()
    {
        $response = $this->actingAs($this->user, 'api')
            ->getJson('/api/flowmaker/smtp/provider-suggestions?provider=gmail');

        $response->assertStatus(200)
            ->assertJson([
                'success' => true,
                'provider' => 'gmail'
            ])
            ->assertJsonStructure([
                'success',
                'provider',
                'configuration' => [
                    'host',
                    'port',
                    'encryption',
                    'notes'
                ]
            ]);
    }

    /** @test */
    public function it_returns_available_providers_when_no_provider_specified()
    {
        $response = $this->actingAs($this->user, 'api')
            ->getJson('/api/flowmaker/smtp/provider-suggestions');

        $response->assertStatus(200)
            ->assertJson([
                'success' => true
            ])
            ->assertJsonStructure([
                'success',
                'available_providers',
                'message'
            ]);
    }

    /** @test */
    public function it_requires_authentication_for_smtp_testing()
    {
        $response = $this->postJson('/api/flowmaker/smtp/test-credentials', [
            'host' => 'smtp.gmail.com',
            'port' => 587,
            'username' => 'test@gmail.com',
            'password' => 'password'
        ]);

        $response->assertStatus(401);
    }

    /** @test */
    public function it_handles_exceptions_gracefully()
    {
        // Test with extremely invalid data that might cause exceptions
        $response = $this->actingAs($this->user, 'api')
            ->postJson('/api/flowmaker/smtp/test-credentials', [
                'host' => str_repeat('a', 1000), // Very long host
                'port' => 999999, // Invalid port
                'username' => 'test@gmail.com',
                'password' => 'password'
            ]);

        // Should handle gracefully, not crash
        $this->assertTrue(in_array($response->status(), [400, 422, 500]));
        $response->assertJsonStructure([
            'success',
            'error'
        ]);
    }
}