<?php

namespace Modules\Flowmaker\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Queue;

class EmailPerformanceService
{
    private const CONNECTION_POOL_KEY = 'smtp_connections';
    private const CONNECTION_TTL = 300; // 5 minutes
    private const MAX_POOL_SIZE = 10;

    /**
     * Get or create SMTP connection from pool
     */
    public function getPooledConnection(array $smtpConfig): ?object
    {
        $connectionKey = $this->generateConnectionKey($smtpConfig);
        $poolKey = self::CONNECTION_POOL_KEY . '_' . $connectionKey;
        
        $connection = Cache::get($poolKey);
        
        if ($connection && $this->isConnectionAlive($connection)) {
            Log::debug('♻️ Reusing pooled SMTP connection', [
                'host' => $smtpConfig['host'],
                'connection_key' => $connectionKey
            ]);
            return $connection;
        }

        return null;
    }

    /**
     * Store SMTP connection in pool
     */
    public function storeConnection(array $smtpConfig, object $connection): void
    {
        $connectionKey = $this->generateConnectionKey($smtpConfig);
        $poolKey = self::CONNECTION_POOL_KEY . '_' . $connectionKey;
        
        // Check pool size limit
        if ($this->getPoolSize() >= self::MAX_POOL_SIZE) {
            $this->evictOldestConnection();
        }

        Cache::put($poolKey, $connection, self::CONNECTION_TTL);
        
        Log::debug('💾 Stored SMTP connection in pool', [
            'host' => $smtpConfig['host'],
            'connection_key' => $connectionKey,
            'ttl' => self::CONNECTION_TTL
        ]);
    }

    /**
     * Queue email for async processing
     */
    public function queueEmail(array $emailData, int $priority = 0): string
    {
        $jobId = uniqid('email_', true);
        
        // Determine queue based on priority
        $queueName = $this->getQueueByPriority($priority);
        
        Queue::pushOn($queueName, new \Modules\Flowmaker\Jobs\SendEmailJob($emailData), $jobId);
        
        Log::info('📤 Email queued for async processing', [
            'job_id' => $jobId,
            'queue' => $queueName,
            'priority' => $priority,
            'to' => $emailData['to'] ?? 'unknown'
        ]);

        return $jobId;
    }

    /**
     * Batch process multiple emails
     */
    public function batchProcessEmails(array $emails): array
    {
        $results = [];
        $batchId = uniqid('batch_', true);
        
        Log::info('📦 Starting batch email processing', [
            'batch_id' => $batchId,
            'email_count' => count($emails)
        ]);

        $startTime = microtime(true);
        
        foreach ($emails as $index => $emailData) {
            try {
                $result = $this->processEmailOptimized($emailData);
                $results[$index] = $result;
                
            } catch (\Exception $e) {
                $results[$index] = [
                    'success' => false,
                    'error' => $e->getMessage()
                ];
            }
        }

        $processingTime = microtime(true) - $startTime;
        
        Log::info('✅ Batch email processing completed', [
            'batch_id' => $batchId,
            'total_emails' => count($emails),
            'successful' => count(array_filter($results, fn($r) => $r['success'] ?? false)),
            'failed' => count(array_filter($results, fn($r) => !($r['success'] ?? true))),
            'processing_time' => round($processingTime, 3) . 's',
            'emails_per_second' => round(count($emails) / $processingTime, 2)
        ]);

        return $results;
    }

    /**
     * Optimize email content for performance
     */
    public function optimizeEmailContent(array $emailData): array
    {
        $optimized = $emailData;

        // Compress HTML content
        if (isset($optimized['body']) && ($optimized['isHtml'] ?? true)) {
            $optimized['body'] = $this->compressHtml($optimized['body']);
        }

        // Optimize attachments
        if (isset($optimized['attachments']) && is_array($optimized['attachments'])) {
            $optimized['attachments'] = $this->optimizeAttachments($optimized['attachments']);
        }

        // Validate and clean recipient lists
        $optimized['to'] = $this->cleanEmailList($optimized['to'] ?? '');
        $optimized['cc'] = $this->cleanEmailList($optimized['cc'] ?? '');
        $optimized['bcc'] = $this->cleanEmailList($optimized['bcc'] ?? '');

        return $optimized;
    }

    /**
     * Monitor email sending performance
     */
    public function recordPerformanceMetrics(array $metrics): void
    {
        $metricsKey = 'email_performance_' . date('Y-m-d-H');
        
        $currentMetrics = Cache::get($metricsKey, [
            'total_emails' => 0,
            'successful_emails' => 0,
            'failed_emails' => 0,
            'total_time' => 0,
            'avg_response_time' => 0,
            'smtp_errors' => []
        ]);

        // Update metrics
        $currentMetrics['total_emails']++;
        
        if ($metrics['success'] ?? false) {
            $currentMetrics['successful_emails']++;
        } else {
            $currentMetrics['failed_emails']++;
            
            // Track error types
            $errorType = $metrics['error_type'] ?? 'unknown';
            $currentMetrics['smtp_errors'][$errorType] = 
                ($currentMetrics['smtp_errors'][$errorType] ?? 0) + 1;
        }

        if (isset($metrics['response_time'])) {
            $currentMetrics['total_time'] += $metrics['response_time'];
            $currentMetrics['avg_response_time'] = 
                $currentMetrics['total_time'] / $currentMetrics['total_emails'];
        }

        Cache::put($metricsKey, $currentMetrics, 3600); // 1 hour
    }

    /**
     * Get performance statistics
     */
    public function getPerformanceStats(int $hours = 24): array
    {
        $stats = [
            'total_emails' => 0,
            'successful_emails' => 0,
            'failed_emails' => 0,
            'success_rate' => 0,
            'avg_response_time' => 0,
            'common_errors' => [],
            'hourly_breakdown' => []
        ];

        for ($i = 0; $i < $hours; $i++) {
            $hour = date('Y-m-d-H', strtotime("-{$i} hours"));
            $metricsKey = 'email_performance_' . $hour;
            $hourlyMetrics = Cache::get($metricsKey, []);

            if (!empty($hourlyMetrics)) {
                $stats['total_emails'] += $hourlyMetrics['total_emails'] ?? 0;
                $stats['successful_emails'] += $hourlyMetrics['successful_emails'] ?? 0;
                $stats['failed_emails'] += $hourlyMetrics['failed_emails'] ?? 0;
                
                $stats['hourly_breakdown'][$hour] = $hourlyMetrics;

                // Aggregate error types
                foreach ($hourlyMetrics['smtp_errors'] ?? [] as $errorType => $count) {
                    $stats['common_errors'][$errorType] = 
                        ($stats['common_errors'][$errorType] ?? 0) + $count;
                }
            }
        }

        // Calculate success rate
        if ($stats['total_emails'] > 0) {
            $stats['success_rate'] = round(
                ($stats['successful_emails'] / $stats['total_emails']) * 100, 
                2
            );
        }

        // Sort errors by frequency
        arsort($stats['common_errors']);

        return $stats;
    }

    /**
     * Process email with performance optimizations
     */
    private function processEmailOptimized(array $emailData): array
    {
        $startTime = microtime(true);
        
        // Optimize content
        $optimizedData = $this->optimizeEmailContent($emailData);
        
        // Use connection pooling if available
        $connection = $this->getPooledConnection($optimizedData['smtp_config'] ?? []);
        
        // Process email (this would integrate with actual email sending logic)
        $result = [
            'success' => true,
            'processing_time' => microtime(true) - $startTime
        ];

        // Record metrics
        $this->recordPerformanceMetrics([
            'success' => $result['success'],
            'response_time' => $result['processing_time']
        ]);

        return $result;
    }

    /**
     * Generate unique key for SMTP connection
     */
    private function generateConnectionKey(array $smtpConfig): string
    {
        $keyData = [
            'host' => $smtpConfig['host'] ?? '',
            'port' => $smtpConfig['port'] ?? '',
            'username' => $smtpConfig['username'] ?? '',
            'encryption' => $smtpConfig['encryption'] ?? ''
        ];

        return md5(serialize($keyData));
    }

    /**
     * Check if SMTP connection is still alive
     */
    private function isConnectionAlive(object $connection): bool
    {
        // This would implement actual connection health check
        // For now, we'll assume connections are valid
        return true;
    }

    /**
     * Get current connection pool size
     */
    private function getPoolSize(): int
    {
        // This would count active connections in the pool
        return 0; // Simplified for now
    }

    /**
     * Evict oldest connection from pool
     */
    private function evictOldestConnection(): void
    {
        // This would remove the oldest connection from the pool
        Log::debug('🗑️ Evicting oldest SMTP connection from pool');
    }

    /**
     * Get queue name based on priority
     */
    private function getQueueByPriority(int $priority): string
    {
        if ($priority >= 10) {
            return 'emails-high';
        } elseif ($priority >= 5) {
            return 'emails-medium';
        } else {
            return 'emails-low';
        }
    }

    /**
     * Compress HTML content
     */
    private function compressHtml(string $html): string
    {
        // Remove unnecessary whitespace and comments
        $html = preg_replace('/<!--.*?-->/s', '', $html);
        $html = preg_replace('/\s+/', ' ', $html);
        $html = trim($html);

        return $html;
    }

    /**
     * Optimize attachments
     */
    private function optimizeAttachments(array $attachments): array
    {
        $optimized = [];
        
        foreach ($attachments as $attachment) {
            if (isset($attachment['path']) && file_exists($attachment['path'])) {
                $fileSize = filesize($attachment['path']);
                
                // Skip very large attachments
                if ($fileSize > 25 * 1024 * 1024) { // 25MB limit
                    Log::warning('📎 Skipping large attachment', [
                        'path' => $attachment['path'],
                        'size' => $fileSize
                    ]);
                    continue;
                }
                
                $optimized[] = $attachment;
            }
        }

        return $optimized;
    }

    /**
     * Clean and validate email list
     */
    private function cleanEmailList(string $emailList): string
    {
        if (empty($emailList)) {
            return '';
        }

        $emails = array_map('trim', explode(',', $emailList));
        $validEmails = array_filter($emails, function($email) {
            return filter_var($email, FILTER_VALIDATE_EMAIL);
        });

        return implode(', ', $validEmails);
    }
}