import React, { useState, useMemo } from 'react';
import { Textarea } from "@/components/ui/textarea";
import { Button } from "@/components/ui/button";
import { Variable } from "lucide-react";
import {
  Command,
  CommandDialog,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import { variables } from '@/data/variablesData';
import { useReactFlow } from '@xyflow/react';

interface VariableTextAreaProps {
  value: string;
  onChange: (value: string) => void;
  placeholder?: string;
  className?: string;
}

type FlowVariable = {
  label: string;
  value: string;
  category: string;
};

export function VariableTextArea({ value, onChange, placeholder, className }: VariableTextAreaProps) {
  const [open, setOpen] = useState(false);
  const { getNodes } = useReactFlow();

  // Extract variables from flow nodes
  const flowVariables: FlowVariable[] = useMemo(() => {
    const nodes = getNodes();
    const dynamicVariables: FlowVariable[] = [];

    nodes.forEach(node => {
      const nodeData = node.data as any;
      const settings = nodeData?.settings || {};
      
      if (node.type === 'openai' && settings.llm?.variableName) {
        dynamicVariables.push({
          label: `Full AI Response for ${settings.llm.variableName}`,
          value: settings.llm.variableName,
          category: 'Flow Variables'
        },{
          label: `AI Response  for ${settings.llm.variableName}`,
          value: settings.llm.variableName + '_message',
          category: 'Flow Variables'
        },{
          label: `Detected Intent for ${settings.llm.variableName}`,
          value: settings.llm.variableName + '_intent',
          category: 'Flow Variables'
        });
      }
      
      if (node.type === 'question' && settings.variableName) {
        dynamicVariables.push({
          label: `User Answer (${nodeData.label || 'Question'})`,
          value: settings.variableName,
          category: 'Flow Variables'
        });
      }
      
      if (node.type === 'http' && settings.http?.responseVar) {
        const base = settings.http.responseVar as string;
        const nodeLabel = nodeData.label || 'HTTP Request';
        dynamicVariables.push(
          { label: `HTTP Response (${nodeLabel})`, value: base, category: 'Flow Variables' },
          { label: `HTTP Status (${nodeLabel})`, value: `${base}.status`, category: 'Flow Variables' },
          { label: `HTTP Data (${nodeLabel})`, value: `${base}.data`, category: 'Flow Variables' },
          { label: `HTTP Headers (${nodeLabel})`, value: `${base}.headers`, category: 'Flow Variables' },
          // Runtime (saved) variants
          { label: `HTTP Status (saved) (${nodeLabel})`, value: `${base}_status`, category: 'Flow Variables' },
          { label: `HTTP Data (saved) (${nodeLabel})`, value: `${base}_data`, category: 'Flow Variables' },
          { label: `HTTP Headers (saved) (${nodeLabel})`, value: `${base}_headers`, category: 'Flow Variables' },
          { label: `HTTP JSON (saved) (${nodeLabel})`, value: `${base}_json`, category: 'Flow Variables' },
          { label: `HTTP Total (saved) (${nodeLabel})`, value: `${base}_total`, category: 'Flow Variables' },
          { label: `HTTP Has Results (saved) (${nodeLabel})`, value: `${base}_has_results`, category: 'Flow Variables' },
          { label: `HTTP Results Count (saved) (${nodeLabel})`, value: `${base}_results_count`, category: 'Flow Variables' },
          // JSON alias for convenience
          { label: `HTTP JSON (${nodeLabel})`, value: `${base}.json`, category: 'Flow Variables' }
        );
      }

      // Support Advanced HTTP node (new_request)
      if (node.type === 'new_request') {
        const base: string | undefined = settings.http?.responseVar || settings.new_request?.responseVar;
        const nodeLabel = nodeData.label || 'Advanced HTTP Request';
        if (base) {
          dynamicVariables.push(
            { label: `HTTP Response (${nodeLabel})`, value: base, category: 'Flow Variables' },
            { label: `HTTP Status (${nodeLabel})`, value: `${base}.status`, category: 'Flow Variables' },
            { label: `HTTP Data (${nodeLabel})`, value: `${base}.data`, category: 'Flow Variables' },
            { label: `HTTP Headers (${nodeLabel})`, value: `${base}.headers`, category: 'Flow Variables' },
            // Runtime (saved) variants
            { label: `HTTP Status (saved) (${nodeLabel})`, value: `${base}_status`, category: 'Flow Variables' },
            { label: `HTTP Data (saved) (${nodeLabel})`, value: `${base}_data`, category: 'Flow Variables' },
            { label: `HTTP Headers (saved) (${nodeLabel})`, value: `${base}_headers`, category: 'Flow Variables' },
            { label: `HTTP JSON (saved) (${nodeLabel})`, value: `${base}_json`, category: 'Flow Variables' },
            { label: `HTTP Total (saved) (${nodeLabel})`, value: `${base}_total`, category: 'Flow Variables' },
            { label: `HTTP Has Results (saved) (${nodeLabel})`, value: `${base}_has_results`, category: 'Flow Variables' },
            { label: `HTTP Results Count (saved) (${nodeLabel})`, value: `${base}_results_count`, category: 'Flow Variables' },
            // JSON alias for convenience
            { label: `HTTP JSON (${nodeLabel})`, value: `${base}.json`, category: 'Flow Variables' }
          );
        }
      }
      
      if (node.type === 'datastore' && settings.variableName) {
        dynamicVariables.push({
          label: `Data Store (${nodeData.label || 'Data Store'})`,
          value: settings.variableName,
          category: 'Flow Variables'
        });
      }
    });

    return dynamicVariables;
  }, [getNodes]);

  // Combine static and dynamic variables
  const allVariables: FlowVariable[] = useMemo(() => {
    return [...(variables as FlowVariable[]), ...flowVariables];
  }, [flowVariables]);

  const insertVariable = (variable: string) => {
    const cursorPosition = (document.activeElement as HTMLTextAreaElement)?.selectionStart || value.length;
    const newValue = value.slice(0, cursorPosition) + `{{${variable}}}` + value.slice(cursorPosition);
    onChange(newValue);
    setOpen(false);
  };

  // Group variables by category
  const groupedVariables: Record<string, FlowVariable[]> = allVariables.reduce(
    (acc: Record<string, FlowVariable[]>, variable: FlowVariable) => {
      const category = variable.category || 'Other';
      if (!acc[category]) {
        acc[category] = [];
      }
      acc[category].push(variable);
      return acc;
    },
    {}
  );

  return (
    <div className="space-y-2">
      <Textarea
        value={value}
        onChange={(e) => onChange(e.target.value)}
        placeholder={placeholder}
        className={`min-h-[100px] ${className || ''}`}
      />
      <div className="flex justify-end">
        <Button
          variant="outline"
          size="sm"
          onClick={() => setOpen(true)}
          className="gap-2"
        >
          <Variable className="h-4 w-4" />
          Variables
        </Button>
      </div>

      <CommandDialog open={open} onOpenChange={setOpen}>
        <Command>
          <CommandInput placeholder="Search variables..." />
          <CommandList>
            <CommandEmpty>No variables found.</CommandEmpty>
            {Object.entries(groupedVariables).map(([category, categoryVariables]) => (
              <CommandGroup key={category} heading={category}>
                {categoryVariables.map((variable) => (
                  <CommandItem
                    key={variable.value}
                    onSelect={() => insertVariable(variable.value)}
                  >
                    {variable.label}
                  </CommandItem>
                ))}
              </CommandGroup>
            ))}
          </CommandList>
        </Command>
      </CommandDialog>
    </div>
  );
}
