<?php

namespace Modules\Flowmaker\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Modules\Flowmaker\Services\SMTPValidator;
use Modules\Flowmaker\Services\EmailErrorHandler;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class SMTPTestController extends Controller
{
    private SMTPValidator $smtpValidator;
    private EmailErrorHandler $errorHandler;

    public function __construct()
    {
        $this->smtpValidator = new SMTPValidator();
        $this->errorHandler = new EmailErrorHandler();
    }

    /**
     * Test SMTP credentials without sending actual emails
     */
    public function testCredentials(Request $request): JsonResponse
    {
        try {
            // Validate request data
            $validator = Validator::make($request->all(), [
                'host' => 'required|string|max:255',
                'port' => 'required|integer|min:1|max:65535',
                'username' => 'required|string|max:255',
                'password' => 'required|string',
                'encryption' => 'nullable|string|in:tls,ssl,none',
                'timeout' => 'nullable|integer|min:5|max:120'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => 'Invalid SMTP configuration',
                    'details' => [
                        'validation_errors' => $validator->errors()->toArray()
                    ]
                ], 422);
            }

            $smtpConfig = [
                'host' => $request->input('host'),
                'port' => (int) $request->input('port'),
                'username' => $request->input('username'),
                'password' => $request->input('password'),
                'encryption' => $request->input('encryption', 'tls'),
                'timeout' => (int) $request->input('timeout', 30)
            ];

            Log::info('🧪 Testing SMTP credentials via API', [
                'host' => $smtpConfig['host'],
                'port' => $smtpConfig['port'],
                'username' => $smtpConfig['username'],
                'encryption' => $smtpConfig['encryption'],
                'user_ip' => $request->ip()
            ]);

            // Validate credentials
            $validationResult = $this->smtpValidator->validateCredentials($smtpConfig);

            if ($validationResult->isValid) {
                Log::info('✅ SMTP credential test successful', [
                    'host' => $smtpConfig['host'],
                    'username' => $smtpConfig['username'],
                    'response_time' => round($validationResult->responseTime, 3) . 's'
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'SMTP credentials are valid and working',
                    'details' => [
                        'host' => $smtpConfig['host'],
                        'port' => $smtpConfig['port'],
                        'username' => $smtpConfig['username'],
                        'encryption' => $smtpConfig['encryption'],
                        'response_time' => round($validationResult->responseTime, 3),
                        'server_response' => $validationResult->serverResponse,
                        'tested_at' => now()->toISOString()
                    ]
                ]);
            } else {
                Log::warning('❌ SMTP credential test failed', [
                    'host' => $smtpConfig['host'],
                    'username' => $smtpConfig['username'],
                    'error' => $validationResult->errorMessage,
                    'response_time' => round($validationResult->responseTime, 3) . 's'
                ]);

                return response()->json([
                    'success' => false,
                    'error' => $validationResult->errorMessage,
                    'details' => [
                        'host' => $smtpConfig['host'],
                        'port' => $smtpConfig['port'],
                        'username' => $smtpConfig['username'],
                        'encryption' => $smtpConfig['encryption'],
                        'response_time' => round($validationResult->responseTime, 3),
                        'server_response' => $validationResult->serverResponse,
                        'tested_at' => now()->toISOString(),
                        'suggestions' => $this->getSuggestionsForError($validationResult->errorMessage)
                    ]
                ], 400);
            }

        } catch (\Exception $e) {
            Log::error('❌ SMTP credential test exception', [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'user_ip' => $request->ip()
            ]);

            return response()->json([
                'success' => false,
                'error' => 'An error occurred while testing SMTP credentials',
                'details' => [
                    'message' => $e->getMessage(),
                    'suggestions' => [
                        'Check all SMTP configuration fields',
                        'Verify your internet connection',
                        'Try again in a few moments'
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Test basic SMTP connection without authentication
     */
    public function testConnection(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'host' => 'required|string|max:255',
                'port' => 'required|integer|min:1|max:65535',
                'timeout' => 'nullable|integer|min:1|max:60'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => 'Invalid connection parameters',
                    'details' => $validator->errors()->toArray()
                ], 422);
            }

            $config = [
                'host' => $request->input('host'),
                'port' => (int) $request->input('port'),
                'timeout' => (int) $request->input('timeout', 10)
            ];

            Log::info('🔗 Testing SMTP connection via API', [
                'host' => $config['host'],
                'port' => $config['port'],
                'timeout' => $config['timeout'],
                'user_ip' => $request->ip()
            ]);

            $connectionResult = $this->smtpValidator->testConnection($config);

            if ($connectionResult) {
                return response()->json([
                    'success' => true,
                    'message' => 'SMTP server is reachable',
                    'details' => [
                        'host' => $config['host'],
                        'port' => $config['port'],
                        'tested_at' => now()->toISOString()
                    ]
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'error' => 'Cannot connect to SMTP server',
                    'details' => [
                        'host' => $config['host'],
                        'port' => $config['port'],
                        'last_error' => $this->smtpValidator->getLastError(),
                        'suggestions' => [
                            'Verify the host address is correct',
                            'Check if the port is correct (587 for TLS, 465 for SSL)',
                            'Ensure your firewall allows outbound connections',
                            'Try a different port number'
                        ]
                    ]
                ], 400);
            }

        } catch (\Exception $e) {
            Log::error('❌ SMTP connection test exception', [
                'error' => $e->getMessage(),
                'user_ip' => $request->ip()
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Connection test failed',
                'details' => ['message' => $e->getMessage()]
            ], 500);
        }
    }

    /**
     * Get configuration suggestions for common email providers
     */
    public function getProviderSuggestions(Request $request): JsonResponse
    {
        $provider = strtolower($request->input('provider', ''));
        
        $suggestions = [
            'gmail' => [
                'host' => 'smtp.gmail.com',
                'port' => 587,
                'encryption' => 'tls',
                'notes' => [
                    'Use your full Gmail address as username',
                    'Enable 2-Factor Authentication and use an App Password',
                    'Generate App Password at: https://myaccount.google.com/apppasswords'
                ]
            ],
            'outlook' => [
                'host' => 'smtp-mail.outlook.com',
                'port' => 587,
                'encryption' => 'tls',
                'notes' => [
                    'Use your full Outlook email as username',
                    'Use your regular Outlook password'
                ]
            ],
            'yahoo' => [
                'host' => 'smtp.mail.yahoo.com',
                'port' => 587,
                'encryption' => 'tls',
                'notes' => [
                    'Use your full Yahoo email as username',
                    'Enable "Less secure app access" or use App Password'
                ]
            ],
            'office365' => [
                'host' => 'smtp.office365.com',
                'port' => 587,
                'encryption' => 'tls',
                'notes' => [
                    'Use your Office 365 email as username',
                    'Use your Office 365 password'
                ]
            ]
        ];

        if (isset($suggestions[$provider])) {
            return response()->json([
                'success' => true,
                'provider' => $provider,
                'configuration' => $suggestions[$provider]
            ]);
        }

        return response()->json([
            'success' => true,
            'available_providers' => array_keys($suggestions),
            'message' => 'Specify a provider parameter to get specific configuration suggestions'
        ]);
    }

    /**
     * Get suggestions based on error message
     */
    private function getSuggestionsForError(string $errorMessage): array
    {
        $errorMessage = strtolower($errorMessage);

        if (strpos($errorMessage, 'authentication') !== false || strpos($errorMessage, '535') !== false) {
            return [
                'Verify your username and password are correct',
                'Check if two-factor authentication is enabled',
                'Try using an app-specific password',
                'Ensure your email provider allows SMTP access'
            ];
        }

        if (strpos($errorMessage, 'connection') !== false || strpos($errorMessage, 'timeout') !== false) {
            return [
                'Check your internet connection',
                'Verify the host and port are correct',
                'Try increasing the timeout value',
                'Check if a firewall is blocking the connection'
            ];
        }

        return [
            'Check all SMTP configuration settings',
            'Verify your email provider\'s SMTP documentation',
            'Contact your email provider for assistance'
        ];
    }
}