<?php

namespace Modules\Flowmaker\Listeners;

use App\Models\Company;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Modules\Flowmaker\Models\Flow;
use Modules\Wpbox\Models\Reply;

class RespondOnMessage
{

    public function handleMessageByContact($event){
        try {
            $contact = $event->message->contact;
            $message = $event->message;
            
            if ($contact->enabled_ai_bot && !$message->bot_has_replied) {
                $this->routeMessage($contact, $message);
            }
        } catch (\Throwable $th) {
            Log::error('Flow routing error', ['error' => $th->getMessage()]);
        }
    }

    private function routeMessage($contact, $message): void
    {
        // Check current active flow
        $currentFlowId = $contact->getCurrentFlowId();
        
        if ($currentFlowId) {
            $flow = Flow::find($currentFlowId);
            if ($flow && $flow->isActive()) {
                Log::info('Continuing current flow', [
                    'contact_id' => $contact->id,
                    'flow_id' => $flow->id,
                    'flow_name' => $flow->name
                ]);
                $flow->processMessage($message);
                return;
            }
        }
        
        // Find matching flow by keywords
        $flows = Flow::where('company_id', $contact->company_id)
                    ->where('is_active', true)
                    ->orderBy('priority', 'desc')
                    ->get();
        
        foreach ($flows as $flow) {
            if ($this->hasKeywordMatch($message, $flow)) {
                Log::info('Switching to keyword-matched flow', [
                    'contact_id' => $contact->id,
                    'flow_id' => $flow->id,
                    'flow_name' => $flow->name,
                    'message' => $message->value
                ]);
                $contact->setCurrentFlowId($flow->id);
                $flow->processMessage($message);
                return;
            }
        }
        
        // Fallback to general flow (priority 0)
        $generalFlow = $flows->where('priority', 0)->first();
        if ($generalFlow) {
            Log::info('Using general flow as fallback', [
                'contact_id' => $contact->id,
                'flow_id' => $generalFlow->id,
                'flow_name' => $generalFlow->name
            ]);
            $generalFlow->processMessage($message);
        } else {
            Log::warning('No general flow found for fallback', [
                'contact_id' => $contact->id,
                'company_id' => $contact->company_id
            ]);
        }
    }

    private function hasKeywordMatch($message, Flow $flow): bool
    {
        // Handle both string and array flow_data
        $flowData = is_string($flow->flow_data) ? json_decode($flow->flow_data, true) : $flow->flow_data;
        if (!$flowData || !isset($flowData['nodes'])) return false;
        
        foreach ($flowData['nodes'] as $node) {
            if ($node['type'] === 'keyword_trigger' && isset($node['data']['keywords'])) {
                foreach ($node['data']['keywords'] as $keyword) {
                    if (str_contains(strtolower($message->value), strtolower($keyword['value']))) {
                        return true;
                    }
                }
            }
        }
        
        return false;
    }

    public function subscribe($events)
    {
        $events->listen(
            'Modules\Wpbox\Events\ContactReplies',
            [RespondOnMessage::class, 'handleMessageByContact']
        );
    }

}
