<?php

namespace Modules\Flowmaker\Services;

use Illuminate\Support\Facades\Log;
use Symfony\Component\Mailer\Transport\Smtp\EsmtpTransport;
use Symfony\Component\Mailer\Transport\Smtp\Stream\SocketStream;
use Exception;

class SMTPValidator
{
    private string $lastError = '';
    private array $lastTestedConfig = [];

    /**
     * Validate SMTP credentials without sending actual emails
     */
    public function validateCredentials(array $smtpConfig): ValidationResult
    {
        $startTime = microtime(true);
        $this->lastTestedConfig = $this->sanitizeConfigForLogging($smtpConfig);
        
        Log::info('🔍 Starting SMTP credential validation', [
            'host' => $smtpConfig['host'] ?? 'unknown',
            'port' => $smtpConfig['port'] ?? 'unknown',
            'username' => $smtpConfig['username'] ?? 'unknown',
            'encryption' => $smtpConfig['encryption'] ?? 'unknown'
        ]);

        try {
            // Validate required fields
            $validationError = $this->validateRequiredFields($smtpConfig);
            if ($validationError) {
                $this->lastError = $validationError;
                return new ValidationResult(
                    false,
                    $validationError,
                    $this->lastTestedConfig,
                    microtime(true) - $startTime,
                    ''
                );
            }

            // Test connection and authentication
            $connectionResult = $this->testConnectionAndAuth($smtpConfig);
            $responseTime = microtime(true) - $startTime;

            if ($connectionResult['success']) {
                Log::info('✅ SMTP validation successful', [
                    'host' => $smtpConfig['host'],
                    'username' => $smtpConfig['username'],
                    'response_time' => round($responseTime, 3) . 's'
                ]);

                return new ValidationResult(
                    true,
                    '',
                    $this->lastTestedConfig,
                    $responseTime,
                    $connectionResult['response']
                );
            } else {
                $this->lastError = $connectionResult['error'];
                
                Log::error('❌ SMTP validation failed', [
                    'host' => $smtpConfig['host'],
                    'username' => $smtpConfig['username'],
                    'error' => $connectionResult['error'],
                    'response_time' => round($responseTime, 3) . 's'
                ]);

                return new ValidationResult(
                    false,
                    $connectionResult['error'],
                    $this->lastTestedConfig,
                    $responseTime,
                    $connectionResult['response']
                );
            }

        } catch (Exception $e) {
            $responseTime = microtime(true) - $startTime;
            $this->lastError = $e->getMessage();
            
            Log::error('❌ SMTP validation exception', [
                'host' => $smtpConfig['host'] ?? 'unknown',
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'response_time' => round($responseTime, 3) . 's'
            ]);

            return new ValidationResult(
                false,
                'Connection failed: ' . $e->getMessage(),
                $this->lastTestedConfig,
                $responseTime,
                ''
            );
        }
    }

    /**
     * Quick connection test without authentication
     */
    public function testConnection(array $smtpConfig): bool
    {
        try {
            $host = $smtpConfig['host'] ?? '';
            $port = $smtpConfig['port'] ?? 587;
            $timeout = $smtpConfig['timeout'] ?? 10;

            if (empty($host)) {
                return false;
            }

            Log::debug('🔗 Testing SMTP connection', [
                'host' => $host,
                'port' => $port,
                'timeout' => $timeout
            ]);

            $socket = @fsockopen($host, $port, $errno, $errstr, $timeout);
            
            if (!$socket) {
                Log::warning('❌ SMTP connection test failed', [
                    'host' => $host,
                    'port' => $port,
                    'error_code' => $errno,
                    'error_message' => $errstr
                ]);
                return false;
            }

            fclose($socket);
            
            Log::debug('✅ SMTP connection test successful', [
                'host' => $host,
                'port' => $port
            ]);
            
            return true;

        } catch (Exception $e) {
            Log::error('❌ SMTP connection test exception', [
                'host' => $smtpConfig['host'] ?? 'unknown',
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get the last error message
     */
    public function getLastError(): string
    {
        return $this->lastError;
    }

    /**
     * Get the last tested configuration (sanitized)
     */
    public function getLastTestedConfig(): array
    {
        return $this->lastTestedConfig;
    }

    /**
     * Validate required SMTP configuration fields
     */
    private function validateRequiredFields(array $smtpConfig): ?string
    {
        $required = ['host', 'port', 'username', 'password'];
        
        foreach ($required as $field) {
            if (empty($smtpConfig[$field])) {
                return "Missing required SMTP field: {$field}";
            }
        }

        // Validate port is numeric
        if (!is_numeric($smtpConfig['port'])) {
            return "SMTP port must be numeric";
        }

        // Validate encryption type
        $validEncryption = ['tls', 'ssl', 'none'];
        $encryption = $smtpConfig['encryption'] ?? 'tls';
        if (!in_array($encryption, $validEncryption)) {
            return "Invalid encryption type. Must be one of: " . implode(', ', $validEncryption);
        }

        return null;
    }

    /**
     * Test SMTP connection and authentication
     */
    private function testConnectionAndAuth(array $smtpConfig): array
    {
        try {
            // Create transport configuration
            $dsn = $this->buildDSN($smtpConfig);
            
            Log::debug('🔧 Creating SMTP transport', [
                'dsn' => $this->sanitizeDSN($dsn),
                'timeout' => $smtpConfig['timeout'] ?? 30
            ]);

            // Create transport
            $transport = new EsmtpTransport(
                $smtpConfig['host'],
                $smtpConfig['port'],
                $smtpConfig['encryption'] === 'ssl'
            );

            // Set authentication
            $transport->setUsername($smtpConfig['username']);
            $transport->setPassword($smtpConfig['password']);

            // Set timeout
            if (isset($smtpConfig['timeout'])) {
                $transport->setTimeout($smtpConfig['timeout']);
            }

            // Test connection by starting the transport
            $transport->start();
            
            // If we get here, authentication was successful
            $response = "SMTP authentication successful";
            
            // Stop the transport
            $transport->stop();

            return [
                'success' => true,
                'error' => '',
                'response' => $response
            ];

        } catch (Exception $e) {
            $errorMessage = $e->getMessage();
            
            // Parse specific error types
            if (strpos($errorMessage, '535') !== false) {
                $errorMessage = 'Authentication failed: Invalid username or password (535)';
            } elseif (strpos($errorMessage, '534') !== false) {
                $errorMessage = 'Authentication failed: Account locked or disabled (534)';
            } elseif (strpos($errorMessage, 'Connection refused') !== false) {
                $errorMessage = 'Connection refused: SMTP server is not reachable';
            } elseif (strpos($errorMessage, 'timeout') !== false) {
                $errorMessage = 'Connection timeout: SMTP server did not respond in time';
            }

            return [
                'success' => false,
                'error' => $errorMessage,
                'response' => $e->getMessage()
            ];
        }
    }

    /**
     * Build DSN string for SMTP transport
     */
    private function buildDSN(array $smtpConfig): string
    {
        $encryption = $smtpConfig['encryption'] ?? 'tls';
        $scheme = ($encryption === 'ssl') ? 'smtps' : 'smtp';
        
        return sprintf(
            '%s://%s:%s@%s:%d',
            $scheme,
            urlencode($smtpConfig['username']),
            urlencode($smtpConfig['password']),
            $smtpConfig['host'],
            $smtpConfig['port']
        );
    }

    /**
     * Sanitize DSN for logging (remove password)
     */
    private function sanitizeDSN(string $dsn): string
    {
        return preg_replace('/(:\/\/[^:]+:)[^@]+(@)/', '$1***$2', $dsn);
    }

    /**
     * Sanitize SMTP config for logging (remove password)
     */
    private function sanitizeConfigForLogging(array $smtpConfig): array
    {
        $sanitized = $smtpConfig;
        if (isset($sanitized['password'])) {
            $sanitized['password'] = '***';
        }
        return $sanitized;
    }
}