<?php

namespace Modules\Flowmaker\Services;

use Illuminate\Support\Facades\Log;
use Exception;

class EmailErrorHandler
{
    /**
     * Handle SMTP-related errors and return standardized response
     */
    public function handleSMTPError(Exception $e, array $config): array
    {
        $errorMessage = $e->getMessage();
        $errorCode = $this->categorizeError($errorMessage);
        
        // Log the error with context
        $this->logSMTPError($e, $config, $errorCode);
        
        // Create user-friendly error response
        return $this->createErrorResponse(
            $this->getUserFriendlyMessage($errorCode, $errorMessage),
            [
                'code' => $errorCode,
                'host' => $config['host'] ?? 'unknown',
                'port' => $config['port'] ?? 'unknown',
                'username' => $config['username'] ?? 'unknown',
                'original_error' => $errorMessage,
                'suggestions' => $this->getSuggestions($errorCode)
            ]
        );
    }

    /**
     * Log authentication failure with detailed information
     */
    public function logAuthenticationFailure(string $host, string $username, string $error): void
    {
        Log::error('🔐 SMTP Authentication Failure', [
            'event' => 'smtp_auth_failed',
            'host' => $host,
            'username' => $username,
            'error' => $error,
            'timestamp' => now()->toISOString(),
            'suggestions' => $this->getSuggestions('SMTP_AUTH_FAILED')
        ]);
    }

    /**
     * Create standardized error response
     */
    public function createErrorResponse(string $message, array $context = []): array
    {
        return [
            'success' => false,
            'error' => $message,
            'details' => $context,
            'timestamp' => now()->toISOString()
        ];
    }

    /**
     * Categorize error based on error message
     */
    private function categorizeError(string $errorMessage): string
    {
        $errorMessage = strtolower($errorMessage);

        // Authentication errors
        if (strpos($errorMessage, '535') !== false || strpos($errorMessage, 'authentication failed') !== false) {
            return 'SMTP_AUTH_FAILED';
        }
        
        if (strpos($errorMessage, '534') !== false || strpos($errorMessage, 'account locked') !== false) {
            return 'SMTP_ACCOUNT_LOCKED';
        }

        // Connection errors
        if (strpos($errorMessage, 'connection refused') !== false) {
            return 'SMTP_CONNECTION_REFUSED';
        }
        
        if (strpos($errorMessage, 'timeout') !== false || strpos($errorMessage, 'timed out') !== false) {
            return 'SMTP_TIMEOUT';
        }
        
        if (strpos($errorMessage, 'could not connect') !== false || strpos($errorMessage, 'network unreachable') !== false) {
            return 'SMTP_NETWORK_ERROR';
        }

        // SSL/TLS errors
        if (strpos($errorMessage, 'ssl') !== false || strpos($errorMessage, 'tls') !== false || strpos($errorMessage, 'certificate') !== false) {
            return 'SMTP_SSL_ERROR';
        }

        // Configuration errors
        if (strpos($errorMessage, 'missing') !== false || strpos($errorMessage, 'required') !== false) {
            return 'SMTP_CONFIG_ERROR';
        }

        // Default to generic SMTP error
        return 'SMTP_GENERIC_ERROR';
    }

    /**
     * Get user-friendly error message based on error code
     */
    private function getUserFriendlyMessage(string $errorCode, string $originalError): string
    {
        switch ($errorCode) {
            case 'SMTP_AUTH_FAILED':
                return 'SMTP authentication failed. Please check your username and password.';
                
            case 'SMTP_ACCOUNT_LOCKED':
                return 'SMTP account is locked or disabled. Please contact your email provider.';
                
            case 'SMTP_CONNECTION_REFUSED':
                return 'SMTP server refused the connection. Please check the host and port settings.';
                
            case 'SMTP_TIMEOUT':
                return 'SMTP connection timed out. The server may be overloaded or unreachable.';
                
            case 'SMTP_NETWORK_ERROR':
                return 'Network error connecting to SMTP server. Please check your internet connection.';
                
            case 'SMTP_SSL_ERROR':
                return 'SSL/TLS connection error. Please verify the encryption settings.';
                
            case 'SMTP_CONFIG_ERROR':
                return 'SMTP configuration error. Please check all required fields are filled.';
                
            default:
                return 'SMTP error occurred: ' . $originalError;
        }
    }

    /**
     * Get suggestions for fixing the error
     */
    private function getSuggestions(string $errorCode): array
    {
        switch ($errorCode) {
            case 'SMTP_AUTH_FAILED':
                return [
                    'Verify your SMTP username and password are correct',
                    'Check if two-factor authentication is enabled and use an app password',
                    'Ensure your email provider allows SMTP access',
                    'Try using your full email address as the username'
                ];
                
            case 'SMTP_ACCOUNT_LOCKED':
                return [
                    'Contact your email provider to unlock the account',
                    'Check if the account has been suspended',
                    'Verify the account is active and in good standing'
                ];
                
            case 'SMTP_CONNECTION_REFUSED':
                return [
                    'Verify the SMTP host address is correct',
                    'Check if the port number is correct (587 for TLS, 465 for SSL)',
                    'Ensure your firewall allows outbound connections on the SMTP port',
                    'Try using a different port (587, 465, or 25)'
                ];
                
            case 'SMTP_TIMEOUT':
                return [
                    'Increase the timeout value in SMTP settings',
                    'Check your internet connection stability',
                    'Try again later as the server may be temporarily overloaded',
                    'Contact your email provider if the issue persists'
                ];
                
            case 'SMTP_NETWORK_ERROR':
                return [
                    'Check your internet connection',
                    'Verify DNS resolution is working',
                    'Try using the server IP address instead of hostname',
                    'Check if a proxy or firewall is blocking the connection'
                ];
                
            case 'SMTP_SSL_ERROR':
                return [
                    'Try changing encryption from TLS to SSL or vice versa',
                    'Verify the port matches the encryption type (587 for TLS, 465 for SSL)',
                    'Check if the server certificate is valid',
                    'Try disabling SSL verification if using a self-signed certificate'
                ];
                
            case 'SMTP_CONFIG_ERROR':
                return [
                    'Ensure all required fields (host, port, username, password) are filled',
                    'Check that the port number is numeric',
                    'Verify the encryption type is valid (TLS, SSL, or none)',
                    'Ensure the username is in the correct format'
                ];
                
            default:
                return [
                    'Check all SMTP configuration settings',
                    'Verify your email provider\'s SMTP documentation',
                    'Try using the default email configuration',
                    'Contact your system administrator for assistance'
                ];
        }
    }

    /**
     * Log SMTP error with full context
     */
    private function logSMTPError(Exception $e, array $config, string $errorCode): void
    {
        // Sanitize config for logging (remove password)
        $sanitizedConfig = $config;
        if (isset($sanitizedConfig['password'])) {
            $sanitizedConfig['password'] = '***';
        }

        Log::error('📧 SMTP Error Occurred', [
            'event' => 'smtp_error',
            'error_code' => $errorCode,
            'error_message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'smtp_config' => $sanitizedConfig,
            'timestamp' => now()->toISOString(),
            'trace' => $e->getTraceAsString()
        ]);
    }

    /**
     * Log successful SMTP operation
     */
    public function logSMTPSuccess(array $config, string $operation = 'send'): void
    {
        Log::info('✅ SMTP Operation Successful', [
            'event' => 'smtp_success',
            'operation' => $operation,
            'host' => $config['host'] ?? 'unknown',
            'port' => $config['port'] ?? 'unknown',
            'username' => $config['username'] ?? 'unknown',
            'timestamp' => now()->toISOString()
        ]);
    }

    /**
     * Create success response for email operations
     */
    public function createSuccessResponse(array $details = []): array
    {
        return [
            'success' => true,
            'message' => 'Email sent successfully',
            'details' => $details,
            'timestamp' => now()->toISOString()
        ];
    }

    /**
     * Log email attempt with configuration used
     */
    public function logEmailAttempt(string $configType, bool $success, string $error = ''): void
    {
        $logData = [
            'event' => 'email_attempt',
            'config_type' => $configType, // 'custom', 'default', 'fallback'
            'success' => $success,
            'timestamp' => now()->toISOString()
        ];

        if (!$success && $error) {
            $logData['error'] = $error;
        }

        if ($success) {
            Log::info('📤 Email Attempt', $logData);
        } else {
            Log::warning('📤 Email Attempt Failed', $logData);
        }
    }
}