import React, { useState, useEffect, useCallback } from 'react';
import { Handle, Position } from '@xyflow/react';
import { 
  FileSpreadsheet, 
  Settings, 
  Info,
  ExternalLink,
  Key,
  // Globe, // COMMENTED: API Key option removed
  Trash2,
  Copy,
  // AlertTriangle // COMMENTED: No longer needed as API Key removed
} from 'lucide-react';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Button } from '@/components/ui/button';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { VariableInput } from '../common/VariableInput';
import { VariableTextArea } from '../common/VariableTextArea';
import { useFlowActions } from '@/hooks/useFlowActions';
import {
  ContextMenu,
  ContextMenuContent,
  ContextMenuItem,
  ContextMenuTrigger,
} from "@/components/ui/context-menu";

interface SheetsNodeProps {
  data: any;
  id: string;
  selected?: boolean;
  isConnectable?: boolean;
}

interface SheetsSettings {
  // operation: 'read' | 'write' | 'update' | 'create_sheet'; // COMMENTED: Now only read/write
  operation: 'read' | 'write'; // UPDATED: Only read and write operations
  spreadsheetId: string;
  range: string;
  authMethod: 'service_account'; // | 'api_key'; // COMMENTED: Only service account now
  serviceAccountPath: string;
  // apiKey: string; // COMMENTED: API Key removed
  responseVar: string;
  dataToWrite: string;
  writeMode: 'append' | 'overwrite';
  // sheetName: string; // COMMENTED: Not needed for read/write only
}

// Service account email for users to share with
const SERVICE_ACCOUNT_EMAIL = "test-node-api@test-sheet-node-project.iam.gserviceaccount.com";

const SheetsNode = ({ data, id, selected, isConnectable }: SheetsNodeProps) => {
  const { deleteNode } = useFlowActions();
  const [nodeSettings, setNodeSettings] = useState<SheetsSettings>(() => {
    const initialSettings: SheetsSettings = {
      operation: data.settings?.operation || 'read',
      spreadsheetId: data.settings?.spreadsheetId || '',
      range: data.settings?.range || 'A:Z',
      authMethod: 'service_account', // data.settings?.authMethod || 'service_account', // UPDATED: Always service account
      serviceAccountPath: data.settings?.serviceAccountPath || 'storage/google-credentials.json',
      // apiKey: data.settings?.apiKey || '', // COMMENTED: API Key removed
      responseVar: data.settings?.responseVar || 'sheets_response',
      dataToWrite: data.settings?.dataToWrite || '',
      writeMode: data.settings?.writeMode || 'append',
      // sheetName: data.settings?.sheetName || 'New Sheet' // COMMENTED: Not needed
    };
    return initialSettings;
  });

  useEffect(() => {
    if (data.settings) {
      Object.keys(nodeSettings).forEach(key => {
        data.settings[key] = nodeSettings[key];
      });
    }
  }, [nodeSettings, data]);

  const updateNodeData = useCallback((key: string, value: any) => {
    setNodeSettings(prev => ({
      ...prev,
      [key]: value
    }));
  }, []);

  const getSpreadsheetIdFromUrl = (url: string) => {
    const match = url.match(/\/spreadsheets\/d\/([a-zA-Z0-9-_]+)/);
    return match ? match[1] : '';
  };

  const handleUrlPaste = (url: string) => {
    const spreadsheetId = getSpreadsheetIdFromUrl(url);
    if (spreadsheetId) {
      updateNodeData('spreadsheetId', spreadsheetId);
    }
  };

  const copyServiceAccountEmail = () => {
    navigator.clipboard.writeText(SERVICE_ACCOUNT_EMAIL);
    // Add toast notification if available
  };

  // COMMENTED: No longer needed as API Key removed
  // Check if current settings would cause an error
  // const hasApiKeyWriteConflict = nodeSettings.authMethod === 'api_key' && 
  //   ['write', 'update', 'create_sheet'].includes(nodeSettings.operation);

  return (
    <ContextMenu>
      <ContextMenuTrigger>
        <div className="bg-white rounded-lg shadow-lg w-[400px] min-h-[600px] flex flex-col">
          <Handle
            type="target"
            position={Position.Left}
            style={{ left: '-4px', background: '#555', zIndex: 50 }}
            isConnectable={isConnectable}
          />

          <div className="flex items-center gap-2 mb-4 pb-2 border-b border-gray-100 px-4 pt-3 bg-green-50">
            <FileSpreadsheet className="h-4 w-4 text-green-600" />
            <div className="font-medium">Google Sheets</div>
            {/* COMMENTED: No warning needed as API Key removed */}
            {/* {hasApiKeyWriteConflict && (
              <AlertTriangle className="h-4 w-4 text-red-500" />
            )} */}
          </div>

          <div className="p-4 space-y-4 flex-1 overflow-y-auto">
            <Tabs defaultValue="configuration" className="w-full">
              <TabsList className="grid w-full grid-cols-2">
                <TabsTrigger value="configuration">Configuration</TabsTrigger>
                <TabsTrigger value="setup">Setup Guide</TabsTrigger>
              </TabsList>

              <TabsContent value="configuration" className="space-y-4">
                {/* COMMENTED: No warning needed as API Key removed */}
                {/* Show warning for API key with write operations */}
                {/* {hasApiKeyWriteConflict && (
                  <Alert className="border-red-200 bg-red-50">
                    <AlertTriangle className="h-4 w-4 text-red-600" />
                    <AlertDescription className="text-red-700">
                      <strong>Configuration Error:</strong> API Key only supports read operations. 
                      Please switch to Service Account or change operation to "Read Data".
                    </AlertDescription>
                  </Alert>
                )} */}

                {/* UPDATED: Operation Selection - Only Read and Write */}
                <div className="space-y-2">
                  <Label htmlFor="operation">Operation</Label>
                  <Select
                    value={nodeSettings.operation}
                    onValueChange={(value) => updateNodeData('operation', value)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="read">Read Data</SelectItem>
                      <SelectItem value="write">Write Data</SelectItem>
                      {/* COMMENTED: Update and Create Sheet removed */}
                      {/* <SelectItem value="update">Update Data</SelectItem>
                      <SelectItem value="create_sheet">Create Sheet</SelectItem> */}
                    </SelectContent>
                  </Select>
                </div>

                {/* COMMENTED: Authentication Method - Only Service Account now */}
                {/* <div className="space-y-2">
                  <Label htmlFor="authMethod">Authentication Method</Label>
                  <Select
                    value={nodeSettings.authMethod}
                    onValueChange={(value) => updateNodeData('authMethod', value)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="service_account">
                        <div className="flex items-center space-x-2">
                          <Key className="w-4 h-4" />
                          <span>Service Account (Full Access)</span>
                        </div>
                      </SelectItem>
                      <SelectItem value="api_key">
                        <div className="flex items-center space-x-2">
                          <Globe className="w-4 h-4" />
                          <span>API Key (Read Only)</span>
                        </div>
                      </SelectItem>
                    </SelectContent>
                  </Select>
                  
                  Info about authentication methods
                  <div className="text-sm text-gray-600">
                    {nodeSettings.authMethod === 'service_account' ? (
                      <p>✅ Can read and write to private sheets. Uses server credentials.</p>
                    ) : (
                      <p>⚠️ Read-only access. Requires public sheets. Enter your API key below.</p>
                    )}
                  </div>
                </div> */}

                {/* UPDATED: Show authentication info without dropdown */}
                <div className="space-y-2">
                  <Label>Authentication</Label>
                  <div className="flex items-center space-x-2 p-3 bg-green-50 rounded-lg border">
                    <Key className="w-4 h-4 text-green-600" />
                    <span className="font-medium text-green-800">Service Account (Full Access)</span>
                  </div>
                  <div className="text-sm text-green-700">
                    ✅ Secure server-side authentication. Can read and write to shared sheets.
                  </div>
                </div>

                {/* Spreadsheet ID */}
                <div className="space-y-2">
                  <Label htmlFor="spreadsheetId">Spreadsheet ID or URL</Label>
                  <div className="flex space-x-2">
                    <VariableInput
                      value={nodeSettings.spreadsheetId}
                      onChange={(value) => updateNodeData('spreadsheetId', value)}
                      placeholder="Enter spreadsheet ID or paste Google Sheets URL"
                      className="flex-1"
                    />
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => {
                        const url = prompt('Paste Google Sheets URL:');
                        if (url) handleUrlPaste(url);
                      }}
                    >
                      <ExternalLink className="w-4 h-4" />
                    </Button>
                  </div>
                </div>

                {/* UPDATED: Range - Always show for both read and write */}
                <div className="space-y-2">
                  <Label htmlFor="range">Range</Label>
                  <VariableInput
                    value={nodeSettings.range}
                    onChange={(value) => updateNodeData('range', value)}
                    placeholder="A1:B10 or Sheet1!A:Z"
                  />
                </div>

                {/* COMMENTED: Sheet Name - Not needed for read/write only */}
                {/* {nodeSettings.operation === 'create_sheet' && (
                  <div className="space-y-2">
                    <Label htmlFor="sheetName">Sheet Name</Label>
                    <VariableInput
                      value={nodeSettings.sheetName}
                      onChange={(value) => updateNodeData('sheetName', value)}
                      placeholder="Enter new sheet name"
                    />
                  </div>
                )} */}

                {/* COMMENTED: Service Account Path - Hidden as requested */}
                {/* {nodeSettings.authMethod === 'service_account' && (
                  <div className="space-y-2">
                    <Label htmlFor="serviceAccountPath">Service Account File Path</Label>
                    <Input
                      value={nodeSettings.serviceAccountPath}
                      onChange={(e) => updateNodeData('serviceAccountPath', e.target.value)}
                      placeholder="storage/google-credentials.json"
                      disabled
                      className="bg-gray-50"
                    />
                    <p className="text-xs text-gray-500">
                      Managed automatically by the system. Upload your service account JSON to the server.
                    </p>
                  </div>
                )} */}

                {/* COMMENTED: API Key - Removed completely */}
                {/* {nodeSettings.authMethod === 'api_key' && (
                  <div className="space-y-2">
                    <Label htmlFor="apiKey">Google API Key</Label>
                    <Input
                      type="password"
                      value={nodeSettings.apiKey}
                      onChange={(e) => updateNodeData('apiKey', e.target.value)}
                      placeholder="Enter your Google API key (AIzaSy...)"
                    />
                    <p className="text-xs text-gray-500">
                      Create an API key in Google Cloud Console and restrict it to Google Sheets API.
                    </p>
                  </div>
                )} */}

                {/* UPDATED: Data to Write - Only for write operations */}
                {nodeSettings.operation === 'write' && (
                  <div className="space-y-2">
                    <Label htmlFor="dataToWrite">Data to Write</Label>
                    <VariableTextArea
                      value={nodeSettings.dataToWrite}
                      onChange={(value) => updateNodeData('dataToWrite', value)}
                      placeholder="Enter JSON array, CSV data, or simple text"
                      rows={4}
                    />
                    <div className="text-sm text-gray-500">
                      <p>Supported formats:</p>
                      <ul className="list-disc list-inside ml-2">
                        <li>JSON: [["Name", "Email"], ["John", "john@example.com"]]</li>
                        <li>CSV: Name,Email\nJohn,john@example.com</li>
                        <li>Simple text: Single value</li>
                      </ul>
                    </div>
                  </div>
                )}

                {/* Write Mode - Only for write operation */}
                {nodeSettings.operation === 'write' && (
                  <div className="space-y-2">
                    <Label htmlFor="writeMode">Write Mode</Label>
                    <Select
                      value={nodeSettings.writeMode}
                      onValueChange={(value) => updateNodeData('writeMode', value)}
                    >
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="append">Append to Sheet</SelectItem>
                        <SelectItem value="overwrite">Overwrite Range</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                )}

                {/* Response Variable */}
                <div className="space-y-2">
                  <Label htmlFor="responseVar">Response Variable</Label>
                  <VariableInput
                    value={nodeSettings.responseVar}
                    onChange={(value) => updateNodeData('responseVar', value)}
                    placeholder="Variable name to store response"
                  />
                </div>
              </TabsContent>

              {/* UPDATED: Simplified Setup Guide for End Users */}
              <TabsContent value="setup" className="space-y-4">
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center space-x-2">
                      <Settings className="w-5 h-5" />
                      <span>Quick Setup for Users</span>
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <Alert>
                      <Info className="h-4 w-4" />
                      <AlertDescription>
                        <strong>Simple 2-step setup:</strong> No technical configuration needed!
                      </AlertDescription>
                    </Alert>

                    <div className="space-y-6">
                      {/* Step 1: Get Spreadsheet ID */}
                      <div>
                        <h4 className="font-semibold mb-3 text-blue-600">📊 Step 1: Get Your Spreadsheet ID</h4>
                        <div className="bg-blue-50 p-4 rounded-lg">
                          <ol className="list-decimal list-inside space-y-2 text-sm">
                            <li>Open your Google Sheet in your browser</li>
                            <li>Look at the URL in your address bar</li>
                            <li>Copy the ID from the URL:</li>
                          </ol>
                          <div className="mt-3 p-3 bg-white rounded border font-mono text-xs break-all">
                            <span className="text-gray-500">https://docs.google.com/spreadsheets/d/</span>
                            <span className="bg-yellow-200 px-1 font-bold">1UGDvT_UFpfu0-N7HB_5zo-Z38Xdl8ZKpO5OaOngAf1s</span>
                            <span className="text-gray-500">/edit</span>
                          </div>
                          <p className="text-xs text-gray-600 mt-2">
                            💡 <strong>Tip:</strong> Or paste the full URL in the field above and click the link button to extract automatically!
                          </p>
                        </div>
                      </div>

                      {/* Step 2: Share with Service Account */}
                      <div>
                        <h4 className="font-semibold mb-3 text-green-600">🔐 Step 2: Share Your Sheet with Our System</h4>
                        <div className="bg-green-50 p-4 rounded-lg">
                          <ol className="list-decimal list-inside space-y-2 text-sm">
                            <li>In your Google Sheet, click the <strong>"Share"</strong> button (top right)</li>
                            <li>In the email field, paste this service account email:</li>
                          </ol>
                          
                          <div className="mt-3 mb-3 p-3 bg-white rounded border">
                            <div className="flex items-center justify-between">
                              <code className="text-xs break-all flex-1 mr-2 font-mono bg-gray-100 px-2 py-1 rounded">
                                {SERVICE_ACCOUNT_EMAIL}
                              </code>
                              <Button 
                                variant="outline" 
                                size="sm" 
                                onClick={copyServiceAccountEmail}
                                className="shrink-0"
                              >
                                <Copy className="w-3 h-3" />
                              </Button>
                            </div>
                          </div>

                          <ol className="list-decimal list-inside space-y-1 text-sm" start={3}>
                            <li>Set permission to:
                              <ul className="list-disc list-inside ml-4 mt-1">
                                <li><strong>"Editor"</strong> - if you want to write data to the sheet</li>
                                <li><strong>"Viewer"</strong> - if you only want to read data from the sheet</li>
                              </ul>
                            </li>
                            <li>Click <strong>"Send"</strong></li>
                            <li>You're done! ✅</li>
                          </ol>
                        </div>
                      </div>

                      {/* Important Notes */}
                      <div className="bg-yellow-50 p-4 rounded-lg border-l-4 border-yellow-400">
                        <h5 className="font-semibold text-yellow-800 mb-2">📝 Important Notes:</h5>
                        <ul className="text-sm text-yellow-700 space-y-1">
                          <li>• Your sheet remains <strong>private</strong> - only you and our system can access it</li>
                          <li>• You can revoke access anytime by removing the service account from sharing</li>
                          <li>• No need to create Google Cloud accounts or API keys</li>
                          <li>• Works with any Google Sheets document you own</li>
                        </ul>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              </TabsContent>
            </Tabs>
          </div>

          <Handle
            type="source"
            position={Position.Right}
            style={{ right: '-4px', background: '#555', zIndex: 50 }}
            isConnectable={isConnectable}
          />
        </div>
      </ContextMenuTrigger>
      <ContextMenuContent>
        <ContextMenuItem
          className="text-red-600 focus:text-red-600 focus:bg-red-100"
          onClick={() => deleteNode(id)}
        >
          <Trash2 className="mr-2 h-4 w-4" />
          Delete
        </ContextMenuItem>
      </ContextMenuContent>
    </ContextMenu>
  );
};

export default SheetsNode;
