import React, { useCallback, useEffect, useMemo, useState } from 'react';
import { Handle, Position, useReactFlow } from '@xyflow/react';
import {
  Globe,
  Trash2,
  Upload,
  X,
  Plus,
  ArrowRight,
  Search,
  Settings,
  FileText,
  Code,
  Database,
  Route,
  Layers,
  ChevronDown,
  ChevronRight,
  Zap,
  Clock,
  RefreshCw,
  Link,
  Filter,
  CheckCircle,
  AlertCircle,
  Info,
  File
} from 'lucide-react';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Button } from '@/components/ui/button';
import { useFlowActions } from '@/hooks/useFlowActions';
import { ContextMenu, ContextMenuContent, ContextMenuItem, ContextMenuTrigger } from '@/components/ui/context-menu';
import { VariableInput } from '@/components/common/VariableInput';

interface NewRequestNodeProps {
  id: string;
  data: any;
}

type HTTPMethod = 'GET' | 'POST' | 'PUT' | 'PATCH' | 'DELETE' | 'HEAD' | 'OPTIONS';
type BodyType = 'json' | 'form' | 'xml' | 'multipart' | 'raw';
type ConditionType = 'status_code' | 'response_contains' | 'response_json' | 'response_header';
type PaginationType = 'none' | 'page' | 'cursor' | 'offset' | 'link_header';
type AuthType = 'none' | 'basic' | 'bearer' | 'api_key';

interface FileUpload {
  id: string;
  name: string;
  file: File;
  fieldName: string;
  size: number;
  type: string;
}

interface Condition {
  id: string;
  type: ConditionType;
  operator: string;
  value: string;
  targetNodeId?: string;
  enabled: boolean;
}

interface PaginationConfig {
  type: PaginationType;
  pageParam: string;
  pageSizeParam: string;
  pageSize: number;
  maxPages: number;
  nextPagePath: string;
  hasMorePath: string;
  enableSendMore: boolean;
  totalCountPath: string;
  currentPagePath: string;
}

interface QueryParam {
  id: string;
  key: string;
  value: string;
  enabled: boolean;
}

interface HeaderParam {
  id: string;
  key: string;
  value: string;
  enabled: boolean;
}

interface AuthConfig {
  type: AuthType;
  basic: {
    username: string;
    password: string;
  };
  bearer: {
    token: string;
  };
  apiKey: {
    key: string;
    value: string;
    location: 'header' | 'query';
  };
}

interface NewRequestSettings {
  method: HTTPMethod;
  url: string;
  headers: HeaderParam[];
  body: string;
  bodyType: BodyType;
  responseVar: string;
  timeout: number;
  retry: number;
  retryDelay: number;
  followRedirects: boolean;
  files: FileUpload[];
  conditions: Condition[];
  enableConditionalRouting: boolean;
  pagination: PaginationConfig;
  queryParams: QueryParam[];
  enableQueryParams: boolean;
  auth: AuthConfig;
  enableAuth: boolean;
  validateSSL: boolean;
  saveResponse: boolean;
  responseFormat: 'auto' | 'json' | 'text' | 'xml';
}

const defaultSettings: NewRequestSettings = {
  method: 'GET',
  url: '',
  headers: [
    { id: '1', key: 'Content-Type', value: 'application/json', enabled: true },
    { id: '2', key: 'Accept', value: 'application/json', enabled: true }
  ],
  body: '{\n  \n}',
  bodyType: 'json',
  responseVar: 'api_response',
  timeout: 30,
  retry: 0,
  retryDelay: 1000,
  followRedirects: true,
  files: [],
  conditions: [],
  enableConditionalRouting: false,
  pagination: {
    type: 'none',
    pageParam: 'page',
    pageSizeParam: 'per_page',
    pageSize: 10,
    maxPages: 5,
    nextPagePath: 'next_page_url',
    hasMorePath: 'has_more',
    enableSendMore: false,
    totalCountPath: 'total',
    currentPagePath: 'current_page'
  },
  queryParams: [],
  enableQueryParams: false,
  auth: {
    type: 'none',
    basic: { username: '', password: '' },
    bearer: { token: '' },
    apiKey: { key: '', value: '', location: 'header' }
  },
  enableAuth: false,
  validateSSL: true,
  saveResponse: true,
  responseFormat: 'auto'
};

const API_PRESETS = {
  woocommerce: {
    name: 'WooCommerce Products',
    method: 'GET' as HTTPMethod,
    url: 'https://namahsoftech.com/demo/test-wordpress/wp-json/wc/v3/products',
    auth: {
      type: 'basic' as AuthType,
      basic: {
        username: 'ck_f18e820e011ffef5be920f2fc560536ec8f0e638',
        password: 'cs_75143af2b76bb899c283570ca3a7a063ffdcf69d'
      },
      bearer: { token: '' },
      apiKey: { key: '', value: '', location: 'header' as const }
    },
    queryParams: [
      { id: '1', key: 'search', value: '{{user_query}}', enabled: true },
      { id: '2', key: 'per_page', value: '5', enabled: true },
      { id: '3', key: 'status', value: 'publish', enabled: true }
    ]
  },
  jsonplaceholder: {
    name: 'JSONPlaceholder Posts',
    method: 'GET' as HTTPMethod,
    url: 'https://jsonplaceholder.typicode.com/posts',
    auth: {
      type: 'none' as AuthType,
      basic: { username: '', password: '' },
      bearer: { token: '' },
      apiKey: { key: '', value: '', location: 'header' as const }
    },
    queryParams: [
      { id: '1', key: '_limit', value: '10', enabled: true }
    ]
  },
  restcountries: {
    name: 'REST Countries',
    method: 'GET' as HTTPMethod,
    url: 'https://restcountries.com/v3.1/all',
    auth: {
      type: 'none' as AuthType,
      basic: { username: '', password: '' },
      bearer: { token: '' },
      apiKey: { key: '', value: '', location: 'header' as const }
    },
    queryParams: [
      { id: '1', key: 'fields', value: 'name,capital,population', enabled: true }
    ]
  }
};

// Normalize legacy/new settings into the new shape
const normalizeHeaders = (input: unknown): HeaderParam[] => {
  try {
    if (Array.isArray(input)) {
      // Already in new format
      return input as HeaderParam[];
    }
    if (typeof input === 'string') {
      const parsed = JSON.parse(input);
      if (parsed && typeof parsed === 'object' && !Array.isArray(parsed)) {
        return Object.entries(parsed as Record<string, string>).map(([key, value], idx) => ({
          id: `${idx + 1}`,
          key,
          value: String(value ?? ''),
          enabled: true
        }));
      }
    }
    if (input && typeof input === 'object') {
      return Object.entries(input as Record<string, string>).map(([key, value], idx) => ({
        id: `${idx + 1}`,
        key,
        value: String(value ?? ''),
        enabled: true
      }));
    }
  } catch (_) {
    // fall through to default
  }
  // default
  return [
    { id: '1', key: 'Content-Type', value: 'application/json', enabled: true },
    { id: '2', key: 'Accept', value: 'application/json', enabled: true }
  ];
};

const normalizeSettings = (raw: any): NewRequestSettings => {
  const base: NewRequestSettings = { ...defaultSettings };
  if (!raw || typeof raw !== 'object') return base;
  const merged: any = { ...base, ...raw };
  merged.headers = normalizeHeaders(merged.headers);
  // Ensure arrays have valid defaults
  if (!Array.isArray(merged.queryParams)) merged.queryParams = [];
  if (!Array.isArray(merged.conditions)) merged.conditions = [];
  if (!Array.isArray(merged.files)) merged.files = [];
  // Body type fallback
  if (!merged.bodyType) merged.bodyType = 'json';
  // Response var fallback
  if (!merged.responseVar) merged.responseVar = base.responseVar;
  // URL fallback
  if (typeof merged.url !== 'string') merged.url = '';
  return merged as NewRequestSettings;
};

const NewRequestNode = ({ id, data }: NewRequestNodeProps) => {
  const { setNodes } = useReactFlow();
  const { deleteNode } = useFlowActions();

  const [settings, setSettings] = useState<NewRequestSettings>(() =>
    normalizeSettings(data?.settings?.new_request)
  );

  const [headersError, setHeadersError] = useState<string | null>(null);
  const [bodyError, setBodyError] = useState<string | null>(null);
  const [urlError, setUrlError] = useState<string | null>(null);
  const [expandedSections, setExpandedSections] = useState({
    basic: true,
    advanced: false,
    files: false,
    routing: false,
    pagination: false,
    queryParams: false,
    headers: false,
    auth: false
  });

  const validateJson = useCallback((jsonString: string) => {
    if (!jsonString || jsonString.trim() === '') return { isValid: true, error: null };
    try {
      JSON.parse(jsonString);
      return { isValid: true, error: null };
    } catch (e) {
      return { isValid: false, error: 'Invalid JSON format' };
    }
  }, []);

  const validateUrl = useCallback((value: string) => {
    if (!value || value.trim() === '') return { isValid: false, error: 'URL is required' };
    try {
      // eslint-disable-next-line no-new
      new URL(value);
      return { isValid: true, error: null };
    } catch {
      return { isValid: false, error: 'Invalid URL format (must include http:// or https://)' };
    }
  }, []);

  useEffect(() => {
    setNodes(nodes =>
      nodes.map(node => {
        if (node.id === id) {
          const currentData = (node.data || {}) as Record<string, unknown>;
          const currentSettings = ((currentData.settings || {}) as Record<string, unknown>);

          const variables: Array<Record<string, unknown>> = [];
          if (settings.responseVar && settings.responseVar.trim()) {
            variables.push({
              name: settings.responseVar,
              type: 'object',
              description: `HTTP ${settings.method} response from ${settings.url || 'API endpoint'}`,
              source: 'new_request'
            });
            variables.push(
              { name: `${settings.responseVar}.status`, type: 'number', description: 'HTTP response status code', source: 'new_request' },
              { name: `${settings.responseVar}.data`, type: 'object', description: 'Response data/body', source: 'new_request' },
              { name: `${settings.responseVar}.headers`, type: 'object', description: 'Response headers', source: 'new_request' }
            );
          }

          return {
            ...node,
            data: {
              ...currentData,
              settings: {
                ...currentSettings,
                new_request: settings,
                // compatibility surface for systems that still read http.* keys
                http: {
                  responseVar: settings.responseVar,
                  method: settings.method,
                  url: settings.url
                }
              },
              variables,
              responseVariable: settings.responseVar
            }
          };
        }
        return node;
      })
    );
  }, [settings, id, setNodes]);

  useEffect(() => {
    const urlValidation = validateUrl(settings.url);
    setUrlError(urlValidation.error);
  }, [settings.url, validateUrl]);

  useEffect(() => {
    const headersJson = settings.headers.reduce((acc, header) => {
      if (header.enabled && header.key && header.value) {
        acc[header.key] = header.value;
      }
      return acc;
    }, {} as Record<string, string>);
    const headersValidation = validateJson(JSON.stringify(headersJson));
    setHeadersError(headersValidation.error);

    if (settings.bodyType === 'json') {
      const bodyValidation = validateJson(settings.body);
      setBodyError(bodyValidation.error);
    } else {
      setBodyError(null);
    }
  }, [settings.headers, settings.body, settings.bodyType, validateJson]);

  const updateSettings = useCallback((updates: Partial<NewRequestSettings>) => {
    setSettings(current => ({
      ...current,
      ...updates
    }));
  }, []);

  const updatePagination = useCallback((updates: Partial<PaginationConfig>) => {
    updateSettings({ pagination: { ...settings.pagination, ...updates } });
  }, [settings.pagination, updateSettings]);

  const updateAuth = useCallback((updates: Partial<AuthConfig>) => {
    updateSettings({ auth: { ...settings.auth, ...updates } });
  }, [settings.auth, updateSettings]);

  const toggleSection = useCallback((section: keyof typeof expandedSections) => {
    setExpandedSections(prev => ({ ...prev, [section]: !prev[section] }));
  }, []);

  const applyPreset = useCallback((presetKey: keyof typeof API_PRESETS) => {
    const preset = API_PRESETS[presetKey];
    updateSettings({
      method: preset.method,
      url: preset.url,
      responseVar: `${presetKey}_response`,
      enableAuth: preset.auth.type !== 'none',
      auth: preset.auth,
      enableQueryParams: preset.queryParams.length > 0,
      queryParams: preset.queryParams,
      headers: [
        { id: '1', key: 'Content-Type', value: 'application/json', enabled: true },
        { id: '2', key: 'Accept', value: 'application/json', enabled: true }
      ]
    });
  }, [updateSettings]);

  const resetToDefaults = useCallback(() => {
    setSettings(defaultSettings);
  }, []);

  const getBodyPlaceholder = useMemo(() => {
    switch (settings.bodyType) {
      case 'json':
        return '{\n  "key": "value",\n  "number": 123\n}';
      case 'form':
        return 'key1=value1&key2=value2';
      case 'xml':
        return '<?xml version="1.0" encoding="UTF-8"?>\n<root>\n  <element>value</element>\n</root>';
      case 'multipart':
        return 'Form data will be automatically handled with files and form fields';
      case 'raw':
        return 'Raw text content';
      default:
        return '';
    }
  }, [settings.bodyType]);

  const getBodyLabel = useMemo(() => {
    switch (settings.bodyType) {
      case 'json': return 'Request Body (JSON)';
      case 'form': return 'Request Body (Form URL Encoded)';
      case 'xml': return 'Request Body (XML)';
      case 'multipart': return 'Request Body (Multipart Form Data)';
      case 'raw': return 'Request Body (Raw Text)';
      default: return 'Request Body';
    }
  }, [settings.bodyType]);

  const getMethodColor = useCallback((method: HTTPMethod) => {
    const colors: Record<string, string> = {
      GET: 'bg-green-100 text-green-800 border-green-200',
      POST: 'bg-blue-100 text-blue-800 border-blue-200',
      PUT: 'bg-yellow-100 text-yellow-800 border-yellow-200',
      PATCH: 'bg-orange-100 text-orange-800 border-orange-200',
      DELETE: 'bg-red-100 text-red-800 border-red-200',
      HEAD: 'bg-purple-100 text-purple-800 border-purple-200',
      OPTIONS: 'bg-gray-100 text-gray-800 border-gray-200'
    };
    return colors[method] || 'bg-gray-100 text-gray-800 border-gray-200';
  }, []);

  const getBodyTypeIcon = useCallback((type: BodyType) => {
    switch (type) {
      case 'json': return <Code className="h-4 w-4" />;
      case 'form': return <Database className="h-4 w-4" />;
      case 'xml': return <FileText className="h-4 w-4" />;
      case 'multipart': return <Upload className="h-4 w-4" />;
      case 'raw': return <FileText className="h-4 w-4" />;
      default: return <FileText className="h-4 w-4" />;
    }
  }, []);

  const handleFileUpload = useCallback((event: React.ChangeEvent<HTMLInputElement>) => {
    const files = event.target.files;
    if (!files) return;
    const newFiles: FileUpload[] = Array.from(files).map(file => ({
      id: Math.random().toString(36).substr(2, 9),
      name: file.name,
      file: file,
      fieldName: file.name.split('.')[0] || 'file',
      size: file.size,
      type: file.type
    }));
    updateSettings({ files: [...settings.files, ...newFiles] });
  }, [settings.files, updateSettings]);

  const removeFile = useCallback((fileId: string) => {
    updateSettings({ files: settings.files.filter(f => f.id !== fileId) });
  }, [settings.files, updateSettings]);

  const updateFileFieldName = useCallback((fileId: string, fieldName: string) => {
    updateSettings({ files: settings.files.map(f => (f.id === fileId ? { ...f, fieldName } : f)) });
  }, [settings.files, updateSettings]);

  const addCondition = useCallback(() => {
    const newCondition: Condition = {
      id: Math.random().toString(36).substr(2, 9),
      type: 'status_code',
      operator: 'equals',
      value: '200',
      enabled: true
    };
    updateSettings({ conditions: [...settings.conditions, newCondition] });
  }, [settings.conditions, updateSettings]);

  const removeCondition = useCallback((conditionId: string) => {
    updateSettings({ conditions: settings.conditions.filter(c => c.id !== conditionId) });
  }, [settings.conditions, updateSettings]);

  const updateCondition = useCallback((conditionId: string, updates: Partial<Condition>) => {
    updateSettings({ conditions: settings.conditions.map(c => (c.id === conditionId ? { ...c, ...updates } : c)) });
  }, [settings.conditions, updateSettings]);

  const addQueryParam = useCallback(() => {
    const newParam: QueryParam = { id: Math.random().toString(36).substr(2, 9), key: '', value: '', enabled: true };
    updateSettings({ queryParams: [...settings.queryParams, newParam] });
  }, [settings.queryParams, updateSettings]);

  const removeQueryParam = useCallback((paramId: string) => {
    updateSettings({ queryParams: settings.queryParams.filter(p => p.id !== paramId) });
  }, [settings.queryParams, updateSettings]);

  const updateQueryParam = useCallback((paramId: string, updates: Partial<QueryParam>) => {
    updateSettings({ queryParams: settings.queryParams.map(p => (p.id === paramId ? { ...p, ...updates } : p)) });
  }, [settings.queryParams, updateSettings]);

  const addHeader = useCallback(() => {
    const newHeader: HeaderParam = { id: Math.random().toString(36).substr(2, 9), key: '', value: '', enabled: true };
    updateSettings({ headers: [...settings.headers, newHeader] });
  }, [settings.headers, updateSettings]);

  const removeHeader = useCallback((headerId: string) => {
    updateSettings({ headers: settings.headers.filter(h => h.id !== headerId) });
  }, [settings.headers, updateSettings]);

  const updateHeader = useCallback((headerId: string, updates: Partial<HeaderParam>) => {
    updateSettings({ headers: settings.headers.map(h => (h.id === headerId ? { ...h, ...updates } : h)) });
  }, [settings.headers, updateSettings]);

  const getOperators = useCallback((type: ConditionType) => {
    const operatorMap: Record<ConditionType, Array<{ value: string; label: string }>> = {
      status_code: [
        { value: 'equals', label: 'Equals' },
        { value: 'not_equals', label: 'Not Equals' },
        { value: 'greater_than', label: 'Greater Than' },
        { value: 'less_than', label: 'Less Than' },
        { value: 'between', label: 'Between' },
        { value: 'in', label: 'In Range' }
      ],
      response_contains: [
        { value: 'contains', label: 'Contains' },
        { value: 'not_contains', label: 'Not Contains' },
        { value: 'starts_with', label: 'Starts With' },
        { value: 'ends_with', label: 'Ends With' },
        { value: 'regex', label: 'Regex Match' }
      ],
      response_json: [
        { value: 'equals', label: 'Equals' },
        { value: 'not_equals', label: 'Not Equals' },
        { value: 'exists', label: 'Exists' },
        { value: 'not_exists', label: 'Not Exists' },
        { value: 'greater_than', label: 'Greater Than' },
        { value: 'less_than', label: 'Less Than' }
      ],
      response_header: [
        { value: 'equals', label: 'Equals' },
        { value: 'contains', label: 'Contains' },
        { value: 'exists', label: 'Exists' }
      ]
    };
    return operatorMap[type] || [];
  }, []);

  const isFormValid = useMemo(() => {
    return Boolean(settings.url && settings.responseVar && !urlError && !headersError && !bodyError);
  }, [settings.url, settings.responseVar, urlError, headersError, bodyError]);

  return (
    <ContextMenu>
      <ContextMenuTrigger>
        <div className="w-[650px] bg-white rounded-xl shadow-lg border border-gray-200 hover:shadow-xl transition-shadow">
          <Handle type="target" position={Position.Left} className="!bg-blue-500 !w-4 !h-4 !rounded-full !border-2 !border-white" />

          <div className="flex items-center gap-3 p-4 bg-gradient-to-r from-blue-50 to-indigo-50 border-b border-gray-200 rounded-t-xl">
            <div className="p-2 bg-blue-100 rounded-lg">
              <Globe className="h-5 w-5 text-blue-600" />
            </div>
            <div className="flex-1">
              <div className="font-semibold text-gray-900">Advanced HTTP Request</div>
              <div className="text-sm text-gray-600">
                {settings.url ? `${settings.method} ${new URL(settings.url).hostname}` : 'Configure your API endpoint'}
              </div>
            </div>
            <div className={`px-3 py-1 rounded-full text-xs font-medium border ${getMethodColor(settings.method)}`}>
              {settings.method}
            </div>
            {!isFormValid && (
              <div className="flex items-center gap-1 px-2 py-1 bg-red-100 text-red-800 text-xs rounded-full">
                <AlertCircle className="h-3 w-3" />
                Issues
              </div>
            )}
          </div>

          <div className="p-4 space-y-4">
            {/* Basic Settings */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="flex items-center gap-2 cursor-pointer mb-3" onClick={() => toggleSection('basic')}>
                {expandedSections.basic ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                <Settings className="h-4 w-4 text-gray-600" />
                <span className="font-medium text-gray-900">Basic Configuration</span>
                {isFormValid && <CheckCircle className="h-4 w-4 text-green-500 ml-auto" />}
              </div>

              {expandedSections.basic && (
                <div className="space-y-4">
                  <div className="grid grid-cols-3 gap-3">
                    <div>
                      <Label className="text-sm font-medium text-gray-700">Method</Label>
                      <Select value={settings.method} onValueChange={(value) => updateSettings({ method: value as HTTPMethod })}>
                        <SelectTrigger className="mt-1">
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="GET">GET</SelectItem>
                          <SelectItem value="POST">POST</SelectItem>
                          <SelectItem value="PUT">PUT</SelectItem>
                          <SelectItem value="PATCH">PATCH</SelectItem>
                          <SelectItem value="DELETE">DELETE</SelectItem>
                          <SelectItem value="HEAD">HEAD</SelectItem>
                          <SelectItem value="OPTIONS">OPTIONS</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                    <div className="col-span-2">
                      <Label className="text-sm font-medium text-gray-700">Response Variable *</Label>
                      <VariableInput
                        placeholder="e.g., api_response"
                        value={settings.responseVar}
                        onChange={(value) => updateSettings({ responseVar: value })}
                        className="mt-1"
                      />
                    </div>
                  </div>

                  <div>
                    <Label className="text-sm font-medium text-gray-700 mb-2 block">Quick Setup Presets</Label>
                    <div className="flex flex-wrap gap-2">
                      <Button variant="outline" size="sm" onClick={() => applyPreset('woocommerce')}>WooCommerce API</Button>
                      <Button variant="outline" size="sm" onClick={() => applyPreset('jsonplaceholder')}>JSONPlaceholder</Button>
                      <Button variant="outline" size="sm" onClick={() => applyPreset('restcountries')}>REST Countries</Button>
                      <Button variant="outline" size="sm" onClick={resetToDefaults} className="text-red-600 hover:text-red-700">Reset All</Button>
                    </div>
                  </div>

                  <div>
                    <Label className="text-sm font-medium text-gray-700">Request URL *</Label>
                    <div className="mt-1 relative">
                      <Link className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                      <VariableInput
                        placeholder="https://api.example.com/endpoint"
                        value={settings.url}
                        onChange={(value) => updateSettings({ url: value })}
                        className="pl-10"
                      />
                      {urlError && (
                        <div className="flex items-center gap-1 text-red-500 text-xs mt-1">
                          <AlertCircle className="h-3 w-3" />
                          {urlError}
                        </div>
                      )}
                    </div>
                  </div>

                  <div className="grid grid-cols-3 gap-4">
                    <div>
                      <Label className="text-sm font-medium text-gray-700 flex items-center gap-2"><Clock className="h-4 w-4" />Timeout (sec)</Label>
                      <Input type="number" min={1} max={300} value={settings.timeout} onChange={e => updateSettings({ timeout: Number(e.target.value) })} className="mt-1" />
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-700 flex items-center gap-2"><RefreshCw className="h-4 w-4" />Retry Count</Label>
                      <Input type="number" min={0} max={10} value={settings.retry} onChange={e => updateSettings({ retry: Number(e.target.value) })} className="mt-1" />
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-700">Response Format</Label>
                      <Select value={settings.responseFormat} onValueChange={(value) => updateSettings({ responseFormat: value as 'auto' | 'json' | 'text' | 'xml' })}>
                        <SelectTrigger className="mt-1">
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="auto">Auto Detect</SelectItem>
                          <SelectItem value="json">JSON</SelectItem>
                          <SelectItem value="text">Text</SelectItem>
                          <SelectItem value="xml">XML</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Advanced Settings */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="flex items-center gap-2 cursor-pointer mb-3" onClick={() => toggleSection('advanced')}>
                {expandedSections.advanced ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                <Zap className="h-4 w-4 text-gray-600" />
                <span className="font-medium text-gray-900">Advanced Settings</span>
              </div>

              {expandedSections.advanced && (
                <div className="space-y-4">
                  {/* Headers list editor */}
                  <div className="bg-white p-3 rounded-lg border">
                    <div className="flex items-center justify-between mb-2">
                      <Label className="text-sm font-medium text-gray-700">Headers</Label>
                      <Button type="button" variant="outline" size="sm" onClick={addHeader}><Plus className="h-4 w-4 mr-1" />Add Header</Button>
                    </div>
                    <div className="space-y-2">
                      {settings.headers.map((h) => (
                        <div key={h.id} className="grid grid-cols-12 gap-2 items-center">
                          <div className="col-span-1 flex items-center justify-center">
                            <input type="checkbox" checked={h.enabled} onChange={e => updateHeader(h.id, { enabled: e.target.checked })} />
                          </div>
                          <div className="col-span-4">
                            <Input placeholder="Header name" value={h.key} onChange={e => updateHeader(h.id, { key: e.target.value })} />
                          </div>
                          <div className="col-span-6">
                            <VariableInput placeholder="Header value" value={h.value} onChange={(value) => updateHeader(h.id, { value })} />
                          </div>
                          <div className="col-span-1 flex justify-end">
                            <Button type="button" variant="ghost" size="sm" className="text-red-500 hover:text-red-700" onClick={() => removeHeader(h.id)}>
                              <X className="h-4 w-4" />
                            </Button>
                          </div>
                        </div>
                      ))}
                    </div>
                    {headersError && (
                      <div className="flex items-center gap-1 text-red-500 text-xs mt-2"><AlertCircle className="h-3 w-3" />{headersError}</div>
                    )}
                  </div>

                  {/* Auth */}
                  <div className="bg-white p-3 rounded-lg border">
                    <div className="flex items-center gap-2 mb-3">
                      <input type="checkbox" id={`enableAuth-${id}`} checked={settings.enableAuth} onChange={e => updateSettings({ enableAuth: e.target.checked })} className="rounded" />
                      <Label htmlFor={`enableAuth-${id}`} className="text-sm font-medium text-gray-700">Use Authentication</Label>
                    </div>
                    {settings.enableAuth && (
                      <div className="space-y-3">
                        <div>
                          <Label className="text-sm font-medium text-gray-700">Auth Type</Label>
                          <Select value={settings.auth.type} onValueChange={(value) => updateAuth({ type: value as AuthType })}>
                            <SelectTrigger className="mt-1"><SelectValue /></SelectTrigger>
                            <SelectContent>
                              <SelectItem value="none">None</SelectItem>
                              <SelectItem value="basic">Basic</SelectItem>
                              <SelectItem value="bearer">Bearer</SelectItem>
                              <SelectItem value="api_key">API Key</SelectItem>
                            </SelectContent>
                          </Select>
                        </div>
                        {settings.auth.type === 'basic' && (
                          <div className="grid grid-cols-2 gap-3">
                            <div>
                              <Label className="text-sm font-medium text-gray-700">Username</Label>
                              <VariableInput value={settings.auth.basic.username} onChange={(value) => updateAuth({ basic: { ...settings.auth.basic, username: value } })} placeholder="username" className="mt-1" />
                            </div>
                            <div>
                              <Label className="text-sm font-medium text-gray-700">Password</Label>
                              <VariableInput value={settings.auth.basic.password} onChange={(value) => updateAuth({ basic: { ...settings.auth.basic, password: value } })} placeholder="password" className="mt-1" />
                            </div>
                          </div>
                        )}
                        {settings.auth.type === 'bearer' && (
                          <div>
                            <Label className="text-sm font-medium text-gray-700">Token</Label>
                            <VariableInput value={settings.auth.bearer.token} onChange={(value) => updateAuth({ bearer: { token: value } })} placeholder="Bearer token" className="mt-1" />
                          </div>
                        )}
                        {settings.auth.type === 'api_key' && (
                          <div className="grid grid-cols-3 gap-3">
                            <div className="col-span-1">
                              <Label className="text-sm font-medium text-gray-700">Key</Label>
                              <Input value={settings.auth.apiKey.key} onChange={e => updateAuth({ apiKey: { ...settings.auth.apiKey, key: e.target.value } })} placeholder="Authorization" className="mt-1" />
                            </div>
                            <div className="col-span-1">
                              <Label className="text-sm font-medium text-gray-700">Value</Label>
                              <VariableInput value={settings.auth.apiKey.value} onChange={(value) => updateAuth({ apiKey: { ...settings.auth.apiKey, value } })} placeholder="token" className="mt-1" />
                            </div>
                            <div className="col-span-1">
                              <Label className="text-sm font-medium text-gray-700">Location</Label>
                              <Select value={settings.auth.apiKey.location} onValueChange={(value) => updateAuth({ apiKey: { ...settings.auth.apiKey, location: value as 'header' | 'query' } })}>
                                <SelectTrigger className="mt-1"><SelectValue /></SelectTrigger>
                                <SelectContent>
                                  <SelectItem value="header">Header</SelectItem>
                                  <SelectItem value="query">Query</SelectItem>
                                </SelectContent>
                              </Select>
                            </div>
                          </div>
                        )}
                      </div>
                    )}
                  </div>

                  <div>
                    <Label className="text-sm font-medium text-gray-700 flex items-center gap-2">{getBodyTypeIcon(settings.bodyType)}Body Type</Label>
                    <Select value={settings.bodyType} onValueChange={(value) => updateSettings({ bodyType: value as BodyType })}>
                      <SelectTrigger className="mt-1"><SelectValue /></SelectTrigger>
                      <SelectContent>
                        <SelectItem value="json">JSON</SelectItem>
                        <SelectItem value="form">Form Data</SelectItem>
                        <SelectItem value="xml">XML</SelectItem>
                        <SelectItem value="multipart">Multipart</SelectItem>
                        <SelectItem value="raw">Raw</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <Label className="text-sm font-medium text-gray-700">{getBodyLabel}</Label>
                    <textarea
                      className="w-full border rounded-lg p-3 text-sm font-mono mt-1 bg-white"
                      rows={5}
                      placeholder={getBodyPlaceholder}
                      value={settings.body}
                      onChange={e => updateSettings({ body: e.target.value })}
                      disabled={settings.bodyType === 'multipart'}
                    />
                    {bodyError && (<div className="flex items-center gap-1 text-red-500 text-xs mt-1"><AlertCircle className="h-3 w-3" />{bodyError}</div>)}
                  </div>
                </div>
              )}
            </div>

            {/* Query Parameters */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="flex items-center gap-2 cursor-pointer mb-3" onClick={() => toggleSection('queryParams')}>
                {expandedSections.queryParams ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                <Search className="h-4 w-4 text-gray-600" />
                <span className="font-medium text-gray-900">Query Parameters</span>
                {settings.queryParams.length > 0 && (<span className="ml-2 px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded-full">{settings.queryParams.length}</span>)}
              </div>
              {expandedSections.queryParams && (
                <div className="space-y-3">
                  {settings.queryParams.map(param => (
                    <div key={param.id} className="flex items-center gap-2 bg-white p-3 rounded-lg border">
                      <VariableInput placeholder="Parameter name" value={param.key} onChange={(value) => updateQueryParam(param.id, { key: value })} className="flex-1" />
                      <VariableInput placeholder="Value" value={param.value} onChange={(value) => updateQueryParam(param.id, { value })} className="flex-1" />
                      <Button type="button" variant="ghost" size="sm" onClick={() => removeQueryParam(param.id)} className="text-red-500 hover:text-red-700"><X className="h-4 w-4" /></Button>
                    </div>
                  ))}
                  <Button type="button" variant="outline" size="sm" onClick={addQueryParam} className="w-full"><Plus className="h-4 w-4 mr-2" />Add Query Parameter</Button>
                </div>
              )}
            </div>

            {/* File Uploads */}
            {settings.bodyType === 'multipart' && (
              <div className="bg-gray-50 rounded-lg p-4">
                <div className="flex items-center gap-2 cursor-pointer mb-3" onClick={() => toggleSection('files')}>
                  {expandedSections.files ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                  <File className="h-4 w-4 text-gray-600" />
                  <span className="font-medium text-gray-900">File Uploads</span>
                  {settings.files.length > 0 && (<span className="ml-2 px-2 py-1 bg-green-100 text-green-800 text-xs rounded-full">{settings.files.length}</span>)}
                </div>
                {expandedSections.files && (
                  <div className="space-y-3">
                    <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 bg-white">
                      <div className="text-center">
                        <Upload className="mx-auto h-8 w-8 text-gray-400 mb-2" />
                        <p className="text-sm text-gray-600 mb-2">Click to upload files or drag and drop</p>
                        <Button type="button" variant="outline" onClick={() => document.getElementById(`file-upload-${id}`)?.click()}>Choose Files</Button>
                        <input id={`file-upload-${id}`} type="file" multiple className="hidden" onChange={handleFileUpload} />
                      </div>
                    </div>
                    {settings.files.length > 0 && (
                      <div className="space-y-2">
                        {settings.files.map(file => (
                          <div key={file.id} className="flex items-center gap-3 p-3 bg-white rounded-lg border">
                            <File className="h-5 w-5 text-blue-500" />
                            <div className="flex-1">
                              <div className="text-sm font-medium text-gray-900">{file.name}</div>
                              <VariableInput placeholder="Field name" value={file.fieldName} onChange={(value) => updateFileFieldName(file.id, value)} className="text-xs mt-1" />
                            </div>
                            <Button type="button" variant="ghost" size="sm" onClick={() => removeFile(file.id)} className="text-red-500 hover:text-red-700"><X className="h-4 w-4" /></Button>
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                )}
              </div>
            )}

            {/* Conditional Routing */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="flex items-center gap-2 cursor-pointer mb-3" onClick={() => toggleSection('routing')}>
                {expandedSections.routing ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                <Route className="h-4 w-4 text-gray-600" />
                <span className="font-medium text-gray-900">Conditional Routing</span>
                {settings.conditions.length > 0 && (<span className="ml-2 px-2 py-1 bg-purple-100 text-purple-800 text-xs rounded-full">{settings.conditions.length}</span>)}
              </div>
              {expandedSections.routing && (
                <div className="space-y-3">
                  {settings.conditions.map(condition => (
                    <div key={condition.id} className="bg-white p-4 rounded-lg border">
                      <div className="flex items-center gap-2 mb-3"><Filter className="h-4 w-4 text-purple-500" /><span className="text-sm font-medium text-gray-900">Condition</span></div>
                      <div className="grid grid-cols-3 gap-2">
                        <Select value={condition.type} onValueChange={(value) => updateCondition(condition.id, { type: value as ConditionType, operator: getOperators(value as ConditionType)[0]?.value || 'equals' })}>
                          <SelectTrigger><SelectValue /></SelectTrigger>
                          <SelectContent>
                            <SelectItem value="status_code">Status Code</SelectItem>
                            <SelectItem value="response_contains">Response Contains</SelectItem>
                            <SelectItem value="response_json">Response JSON</SelectItem>
                            <SelectItem value="response_header">Response Header</SelectItem>
                          </SelectContent>
                        </Select>

                        <Select value={condition.operator} onValueChange={(value) => updateCondition(condition.id, { operator: value })}>
                          <SelectTrigger><SelectValue /></SelectTrigger>
                          <SelectContent>
                            {getOperators(condition.type).map(op => (
                              <SelectItem key={op.value} value={op.value}>{op.label}</SelectItem>
                            ))}
                          </SelectContent>
                        </Select>

                        <div className="flex gap-2">
                          <VariableInput placeholder="Value" value={condition.value} onChange={(value) => updateCondition(condition.id, { value })} className="flex-1" />
                          <Button type="button" variant="ghost" size="sm" onClick={() => removeCondition(condition.id)} className="text-red-500 hover:text-red-700"><X className="h-4 w-4" /></Button>
                        </div>
                      </div>
                    </div>
                  ))}
                  <Button type="button" variant="outline" size="sm" onClick={addCondition} className="w-full"><Plus className="h-4 w-4 mr-2" />Add Condition</Button>
                </div>
              )}
            </div>

            {/* Pagination */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="flex items-center gap-2 cursor-pointer mb-3" onClick={() => toggleSection('pagination')}>
                {expandedSections.pagination ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                <Layers className="h-4 w-4 text-gray-600" />
                <span className="font-medium text-gray-900">Pagination & Send More</span>
                {settings.pagination.enableSendMore && (<span className="ml-2 px-2 py-1 bg-orange-100 text-orange-800 text-xs rounded-full">Active</span>)}
              </div>
              {expandedSections.pagination && (
                <div className="space-y-4">
                  <div className="bg-white p-4 rounded-lg border">
                    <div className="flex items-center gap-3 mb-4">
                      <div className="flex items-center gap-2">
                        <input type="checkbox" id={`pagination-${id}`} checked={settings.pagination.enableSendMore} onChange={(e) => updatePagination({ enableSendMore: e.target.checked })} className="rounded" />
                        <Label htmlFor={`pagination-${id}`} className="text-sm font-medium text-gray-700">Enable Pagination / Send More</Label>
                      </div>
                      {settings.pagination.enableSendMore && (
                        <div className="flex items-center gap-2 px-2 py-1 bg-green-100 text-green-800 text-xs rounded-full">
                          <CheckCircle className="h-3 w-3" /> Enabled
                        </div>
                      )}
                    </div>

                    {settings.pagination.enableSendMore && (
                      <div className="space-y-4">
                        <div>
                          <Label className="text-sm font-medium text-gray-700 flex items-center gap-2"><Database className="h-4 w-4" />Pagination Type</Label>
                          <Select value={settings.pagination.type} onValueChange={(value) => updatePagination({ type: value as PaginationType })}>
                            <SelectTrigger className="mt-1"><SelectValue /></SelectTrigger>
                            <SelectContent>
                              <SelectItem value="none">None</SelectItem>
                              <SelectItem value="page">Page Number</SelectItem>
                              <SelectItem value="cursor">Cursor-based</SelectItem>
                              <SelectItem value="offset">Offset-based</SelectItem>
                              <SelectItem value="link_header">Link Header</SelectItem>
                            </SelectContent>
                          </Select>
                        </div>

                        {settings.pagination.type !== 'none' && (
                          <div className="space-y-4">
                            <div className="grid grid-cols-2 gap-3">
                              <div>
                                <Label className="text-sm font-medium text-gray-700">Page Parameter</Label>
                                <VariableInput placeholder="page" value={settings.pagination.pageParam} onChange={(value) => updatePagination({ pageParam: value })} className="mt-1" />
                              </div>
                              <div>
                                <Label className="text-sm font-medium text-gray-700">Page Size Parameter</Label>
                                <VariableInput placeholder="per_page" value={settings.pagination.pageSizeParam} onChange={(value) => updatePagination({ pageSizeParam: value })} className="mt-1" />
                              </div>
                            </div>
                            <div className="grid grid-cols-2 gap-3">
                              <div>
                                <Label className="text-sm font-medium text-gray-700">Page Size</Label>
                                <Input type="number" min={1} max={100} value={settings.pagination.pageSize} onChange={(e) => updatePagination({ pageSize: Number(e.target.value) })} className="mt-1" />
                              </div>
                              <div>
                                <Label className="text-sm font-medium text-gray-700">Max Pages</Label>
                                <Input type="number" min={1} max={20} value={settings.pagination.maxPages} onChange={(e) => updatePagination({ maxPages: Number(e.target.value) })} className="mt-1" />
                              </div>
                            </div>
                            <div>
                              <Label className="text-sm font-medium text-gray-700 flex items-center gap-2"><ArrowRight className="h-4 w-4" />Next Page Path (JSON path)</Label>
                              <VariableInput placeholder="next_page_url" value={settings.pagination.nextPagePath} onChange={(value) => updatePagination({ nextPagePath: value })} className="mt-1" />
                            </div>
                            <div>
                              <Label className="text-sm font-medium text-gray-700 flex items-center gap-2"><Info className="h-4 w-4" />Has More Path (JSON path)</Label>
                              <VariableInput placeholder="has_more" value={settings.pagination.hasMorePath} onChange={(value) => updatePagination({ hasMorePath: value })} className="mt-1" />
                            </div>
                          </div>
                        )}
                      </div>
                    )}
                  </div>
                </div>
              )}
            </div>
          </div>

          <Handle type="source" position={Position.Right} className="!bg-green-500 !w-4 !h-4 !rounded-full" />
        </div>
      </ContextMenuTrigger>
      <ContextMenuContent>
        <ContextMenuItem className="text-red-600 focus:text-red-600 focus:bg-red-100" onClick={() => deleteNode(id)}>
          <Trash2 className="mr-2 h-4 w-4" />
          Delete Node
        </ContextMenuItem>
      </ContextMenuContent>
    </ContextMenu>
  );
};

export default NewRequestNode;


