import {
  ReactFlow,
  Background,
  useNodesState,
  useEdgesState,
  addEdge,
  Connection,
  Edge,
  Node,
} from '@xyflow/react';
import '@xyflow/react/dist/style.css';
import { Button } from "@/components/ui/button";
import { Save, ArrowLeft } from "lucide-react";
import { useToast } from "@/components/ui/use-toast";
import { useCallback, useState, useRef, useEffect } from 'react';
import ActionPanel from './ActionPanel';
import { nodeTypes } from '@/config/nodeTypes';
import DataSidebar from './DataSidebar';

interface FlowCanvasProps {
  flowId?: string;
}

const flow_data = window.data?.flow?.flow_data || "{}";
const initialNodes: Node[] = JSON.parse(flow_data)?.nodes || [
];

const initialEdges: Edge[] = JSON.parse(flow_data)?.edges || [];

// Default edge options for animated black edges
const defaultEdgeOptions = {
  animated: true,
  style: {
    stroke: '#000000',
  },
};

console.log('Initial nodes:', initialNodes);
console.log('Initial edges:', initialEdges);
console.log('========= Window data ========');
console.log(window.data);


const FlowCanvas = ({ flowId = '1' }: FlowCanvasProps) => {
  const [nodes, setNodes, onNodesChange] = useNodesState(initialNodes);
  const [edges, setEdges, onEdgesChange] = useEdgesState(initialEdges);
  const [dataDrawerOpen, setDataDrawerOpen] = useState(false);
  const { toast } = useToast();
  const fileInputRef = useRef<HTMLInputElement>(null);

  const saveButtonRef = useRef<HTMLButtonElement>(null);

  useEffect(() => {
    const intervalId = setInterval(() => {
      if (saveButtonRef.current) {
        saveButtonRef.current.click();
      }
    }, 600000); // 10 minutes = 600000

    return () => clearInterval(intervalId);
  }, []);


  const onConnect = useCallback(
    (params: Connection | Edge) => {
      const newEdge = {
        ...params,
        id: `e${Date.now()}`,
      };
      return setEdges((eds) => addEdge(newEdge, eds));
    },
    [setEdges],
  );

  const handleSave = async () => {
    const nodesWithData = nodes.map(node => ({
      ...node,
      data: {
        ...node.data,
        settings: node.data?.settings || {},
      },
    }));

    const flowData = {
      nodes: nodesWithData,
      edges,
    };
    
    console.log('Saving flow:', flowData);
    console.log('Flow ID:',window.data.flow.id);
    
    try {
      const response = await fetch(`/flowmaker/update/${window.data.flow.id}`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(flowData),
      });

      if (!response.ok) {
        throw new Error('Failed to save flow');
      }

      toast({
        title: "Flow saved",
        description: "Your flow has been saved successfully.",
      });
    } catch (error) {
      console.error('Error saving flow:', error);
      toast({
        title: "Save failed",
        description: "Failed to save flow. Please try again.",
        variant: "destructive",
      });
    }
  };

  const handlePopoverIndexChange = (index: number | null) => {
    // No longer automatically opening the data sidebar when Data popover is shown
  };
  
  const handleOpenDataSidebar = () => {
    setDataDrawerOpen(true);
  };

  const handleBackClick = () => {
    window.location.href = "/flows";
  };

  const handleImportClick = () => {
    if (fileInputRef.current) fileInputRef.current.value = '';
    fileInputRef.current?.click();
  };

  const handleFileSelected = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = event.target.files;
    if (!files || files.length === 0) return;
    const file = files[0];
    if (file.type !== 'application/json' && !file.name.endsWith('.json')) {
      toast({ title: 'Invalid file', description: 'Please select a .json file.', variant: 'destructive' });
      return;
    }
    if (file.size > 10 * 1024 * 1024) {
      toast({ title: 'File too large', description: 'Max size is 10MB.', variant: 'destructive' });
      return;
    }
    const formData = new FormData();
    formData.append('file', file);
    // Add flow_id to FormData
    const currentFlowId = window.data?.flow?.id || flowId;
    if (!currentFlowId) {
      toast({ title: 'No flow selected', description: 'Cannot determine which flow to update.', variant: 'destructive' });
      return;
    }
    formData.append('flow_id', currentFlowId);
    try {
      const response = await fetch('/flowmaker/import', {
        method: 'POST',
        body: formData,
      });
      const data = await response.json();
      if (response.ok && data.status === 'ok') {
        toast({ title: 'Imported', description: 'Flow imported successfully.' });
        const currentId = window.data?.flow?.id || currentFlowId;
        if (String(currentId) === String(data.flow_id)) {
          // Same flow; hard reload to reflect new nodes/edges immediately
          window.location.reload();
        } else {
          // Different flow; navigate to the imported flow editor
          window.location.href = `/flowmaker/edit/${data.flow_id}`;
        }
      } else {
        toast({ title: 'Import failed', description: data.error || 'Could not import flow.', variant: 'destructive' });
      }
    } catch (e) {
      toast({ title: 'Import failed', description: 'Could not import flow.', variant: 'destructive' });
    }
  };

  return (
    <div className="flex h-screen bg-[#F1F0FB] relative">
      {/* Left Sidebar */}
      <div className="absolute left-0 top-0 m-2.5 z-10">
        <div className="w-20 flex flex-col items-center py-4 space-y-6">
          <div className="flex flex-col items-center space-y-6">
            <ActionPanel 
              onImportClick={() => {}}
              onPopoverIndexChange={handlePopoverIndexChange}
              onOpenDataSidebar={handleOpenDataSidebar}
            />
          </div>
        </div>
      </div>

      {/* Main Flow Area */}
      <div className="flex-1 relative">
        {/* Back and Save Buttons */}
        <div className="absolute top-4 right-4 z-10 flex space-x-2">
          <Button 
            onClick={handleBackClick}
            variant="outline"
            size="default"
            className="gap-2 bg-white"
          >
            <ArrowLeft className="h-4 w-4" />
            Back
          </Button>
          <Button 
            ref={saveButtonRef}
            onClick={handleSave}
            variant="outline"
            size="default"
            className="gap-2 bg-white"
          >
            <Save className="h-4 w-4" />
            Save
          </Button>
          {/* Export Button */}
          <Button
            onClick={async () => {
              try {
                const currentFlowId = window.data?.flow?.id || flowId;
                const response = await fetch(`/flowmaker/export/${currentFlowId}`);
                if (!response.ok) throw new Error('Export failed');
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `flowmaker_export_${currentFlowId}.json`;
                document.body.appendChild(a);
                a.click();
                a.remove();
                window.URL.revokeObjectURL(url);
                toast({ title: 'Exported', description: 'Flow exported as JSON.' });
              } catch (e) {
                toast({ title: 'Export failed', description: 'Could not export flow.', variant: 'destructive' });
              }
            }}
            variant="outline"
            size="default"
            className="gap-2 bg-white"
          >
            Export
          </Button>
          {/* Import Button */}
          <Button
            onClick={handleImportClick}
            variant="outline"
            size="default"
            className="gap-2 bg-white"
          >
            Import
          </Button>
          <input
            ref={fileInputRef}
            type="file"
            accept=".json,application/json"
            style={{ display: 'none' }}
            onChange={handleFileSelected}
          />
        </div>

        <ReactFlow
          nodes={nodes}
          edges={edges}
          onNodesChange={onNodesChange}
          onEdgesChange={onEdgesChange}
          onConnect={onConnect}
          nodeTypes={nodeTypes}
          defaultEdgeOptions={defaultEdgeOptions}
          minZoom={0.1}
          maxZoom={4}
          fitView
        >
          <Background />
        </ReactFlow>

        {/* Data Sidebar */}
        <DataSidebar 
          open={dataDrawerOpen}
          onOpenChange={setDataDrawerOpen}
        />
      </div>
    </div>
  );
};

export default FlowCanvas;
