import React, { useCallback, useState, useEffect } from 'react';
import { Handle, Position } from '@xyflow/react';
import { MessageSquare, Plus, X, Trash2, AlertCircle } from 'lucide-react';
import { Label } from "@/components/ui/label";
import { Button } from "@/components/ui/button";
import { NodeData } from '@/types/flow';
import { VariableTextArea } from '../common/VariableTextArea';
import { VariableInput } from '../common/VariableInput';
import { useFlowActions } from "@/hooks/useFlowActions";
import {
  ContextMenu,
  ContextMenuContent,
  ContextMenuItem,
  ContextMenuTrigger,
} from "@/components/ui/context-menu";

interface QuickRepliesNodeProps {
  data: NodeData;
  id: string;
  selected?: boolean;
  isConnectable?: boolean;
}

interface NodeSettings {
  header: string;
  body: string;
  footer: string;
  activeButtons: number;
  listButtonName: string;
  [key: string]: string | number; // Allow dynamic button keys
}

// WhatsApp character limits (Official API Documentation)
const WHATSAPP_LIMITS = {
  MESSAGE_TEXT: 1024,           // Official limit for interactive message text
  QUICK_REPLY_BUTTON: 20,       // Reply button text limit
  LIST_BUTTON: 20,              // List button text limit  
  LIST_ITEM_TITLE: 24,          // List item title limit
  LIST_ITEM_DESCRIPTION: 72,    // List item description limit
  MAX_QUICK_REPLY_BUTTONS: 3,   // Maximum reply buttons allowed
  MAX_LIST_ITEMS: 10,           // Maximum list items allowed
  MAX_TOTAL_BUTTONS: 10         // Maximum total buttons (3 reply + 7 list)
};

const QuickRepliesNode = ({ data, id, selected, isConnectable }: QuickRepliesNodeProps) => {
  const { deleteNode } = useFlowActions();
  const [nodeSettings, setNodeSettings] = useState<NodeSettings>(() => {
    const initialSettings: NodeSettings = {
      header: data.settings?.header || '',
      body: data.settings?.body || '',
      footer: data.settings?.footer || '',
      activeButtons: data.settings?.activeButtons || 1,
      listButtonName: data.settings?.listButtonName || '',
    };

    // Add button fields
    for (let i = 1; i <= WHATSAPP_LIMITS.MAX_TOTAL_BUTTONS; i++) {
      initialSettings[`button${i}`] = data.settings?.[`button${i}`] || '';
    }

    return initialSettings;
  });

  useEffect(() => {
    if (!data.settings) return;
    Object.keys(nodeSettings).forEach(key => {
      (data.settings as any)[key] = nodeSettings[key];
    });
  }, [nodeSettings, data]);

  const updateNodeData = useCallback((key: string, value: any) => {
    setNodeSettings(prev => ({
      ...prev,
      [key]: value
    }));
  }, []);

  const addButton = useCallback(() => {
    const currentLimit = getButtonLimit();
    if (nodeSettings.activeButtons < currentLimit) {
      updateNodeData('activeButtons', nodeSettings.activeButtons + 1);
    }
  }, [nodeSettings.activeButtons, updateNodeData]);

  const removeButton = useCallback((buttonIndex: number) => {
    if (nodeSettings.activeButtons > 1) {
      const newSettings = { ...nodeSettings };
      for (let i = buttonIndex; i < WHATSAPP_LIMITS.MAX_TOTAL_BUTTONS; i++) {
        newSettings[`button${i}`] = nodeSettings[`button${i + 1}`] || '';
      }
      newSettings[`button${WHATSAPP_LIMITS.MAX_TOTAL_BUTTONS}`] = '';
      newSettings.activeButtons = nodeSettings.activeButtons - 1;
      setNodeSettings(newSettings);
    }
  }, [nodeSettings]);

  const isListMode = nodeSettings.activeButtons > WHATSAPP_LIMITS.MAX_QUICK_REPLY_BUTTONS;

  // Validation functions
  const getCharacterCount = (text: string) => text.length;
  
  const getButtonLimit = () => {
    return isListMode ? WHATSAPP_LIMITS.MAX_LIST_ITEMS : WHATSAPP_LIMITS.MAX_QUICK_REPLY_BUTTONS;
  };

  const getButtonCharLimit = () => {
    return isListMode ? WHATSAPP_LIMITS.LIST_ITEM_TITLE : WHATSAPP_LIMITS.QUICK_REPLY_BUTTON;
  };

  const validateButtonText = (text: string) => {
    const limit = getButtonCharLimit();
    return text.length <= limit;
  };

  const validateMessageText = (text: string) => {
    return text.length <= WHATSAPP_LIMITS.MESSAGE_TEXT;
  };

  const validateListButtonName = (text: string) => {
    return text.length <= WHATSAPP_LIMITS.LIST_BUTTON;
  };

  return (
    <ContextMenu>
      <ContextMenuTrigger>
        <div className="bg-white rounded-lg shadow-lg w-[300px]">
          <Handle
            type="target"
            position={Position.Left}
            style={{ left: '-4px', background: '#555', zIndex: 50 }}
            isConnectable={isConnectable}
          />

          <div className="flex items-center gap-2 mb-4 pb-2 border-b border-gray-100 px-4 pt-3 bg-gray-50">
            <MessageSquare className="h-4 w-4 text-indigo-600" />
            <div className="font-medium">Quick Replies Message</div>
          </div>

          <div className="p-4 space-y-4">
            {/* WhatsApp Limitations Info */}
            <div className="bg-blue-50 border border-blue-200 rounded-md p-3">
              <div className="text-sm text-blue-800 space-y-1">
                <div className="font-medium">WhatsApp Limitations:</div>
                <div>• Message text: {WHATSAPP_LIMITS.MESSAGE_TEXT} characters max</div>
                <div>• Quick reply buttons: {WHATSAPP_LIMITS.MAX_QUICK_REPLY_BUTTONS} buttons, {WHATSAPP_LIMITS.QUICK_REPLY_BUTTON} chars each</div>
                <div>• List items: {WHATSAPP_LIMITS.MAX_LIST_ITEMS} items, {WHATSAPP_LIMITS.LIST_ITEM_TITLE} chars title, {WHATSAPP_LIMITS.LIST_ITEM_DESCRIPTION} chars description</div>
              </div>
            </div>

            {isListMode && (
              <div className="bg-yellow-50 border border-yellow-200 rounded-md p-3 flex items-start gap-2">
                <AlertCircle className="h-4 w-4 text-yellow-600 mt-0.5" />
                <div className="text-sm text-yellow-700">
                  With more than {WHATSAPP_LIMITS.MAX_QUICK_REPLY_BUTTONS} buttons, this will be sent as a List Message
                </div>
              </div>
            )}

            <div className="space-y-2">
              <div className="flex justify-between items-center">
                <Label htmlFor={`${id}-header`}>Header</Label>
                <span className={`text-xs ${getCharacterCount(nodeSettings.header) > WHATSAPP_LIMITS.MESSAGE_TEXT ? 'text-red-500' : 'text-gray-500'}`}>
                  {getCharacterCount(nodeSettings.header)}/{WHATSAPP_LIMITS.MESSAGE_TEXT}
                </span>
              </div>
              <VariableInput
                id={`${id}-header`}
                placeholder="Enter header text"
                value={nodeSettings.header}
                onChange={(value) => updateNodeData('header', value)}
                className={!validateMessageText(nodeSettings.header) ? 'border-red-500' : ''}
              />
              {!validateMessageText(nodeSettings.header) && (
                <p className="text-xs text-red-500">Header text exceeds WhatsApp limit</p>
              )}
            </div>

            <div className="space-y-2">
              <div className="flex justify-between items-center">
                <Label htmlFor={`${id}-body`}>Body</Label>
                <span className={`text-xs ${getCharacterCount(nodeSettings.body) > WHATSAPP_LIMITS.MESSAGE_TEXT ? 'text-red-500' : 'text-gray-500'}`}>
                  {getCharacterCount(nodeSettings.body)}/{WHATSAPP_LIMITS.MESSAGE_TEXT}
                </span>
              </div>
              <VariableTextArea
                value={nodeSettings.body}
                onChange={(value) => updateNodeData('body', value)}
                placeholder="Enter message body"
                className={!validateMessageText(nodeSettings.body) ? 'border-red-500' : ''}
              />
              {!validateMessageText(nodeSettings.body) && (
                <p className="text-xs text-red-500">Body text exceeds WhatsApp limit</p>
              )}
            </div>

            <div className="space-y-2">
              <div className="flex justify-between items-center">
                <Label htmlFor={`${id}-footer`}>Footer</Label>
                <span className={`text-xs ${getCharacterCount(nodeSettings.footer) > WHATSAPP_LIMITS.MESSAGE_TEXT ? 'text-red-500' : 'text-gray-500'}`}>
                  {getCharacterCount(nodeSettings.footer)}/{WHATSAPP_LIMITS.MESSAGE_TEXT}
                </span>
              </div>
              <VariableInput
                id={`${id}-footer`}
                placeholder="Enter footer text"
                value={nodeSettings.footer}
                onChange={(value) => updateNodeData('footer', value)}
                className={!validateMessageText(nodeSettings.footer) ? 'border-red-500' : ''}
              />
              {!validateMessageText(nodeSettings.footer) && (
                <p className="text-xs text-red-500">Footer text exceeds WhatsApp limit</p>
              )}
            </div>

            {isListMode && (
              <div className="space-y-2">
                <div className="flex justify-between items-center">
                  <Label htmlFor={`${id}-list-button`}>List Button Name</Label>
                  <span className={`text-xs ${getCharacterCount(nodeSettings.listButtonName) > WHATSAPP_LIMITS.LIST_BUTTON ? 'text-red-500' : 'text-gray-500'}`}>
                    {getCharacterCount(nodeSettings.listButtonName)}/{WHATSAPP_LIMITS.LIST_BUTTON}
                  </span>
                </div>
                <VariableInput
                  id={`${id}-list-button`}
                  placeholder="Enter list button name"
                  value={nodeSettings.listButtonName}
                  onChange={(value) => updateNodeData('listButtonName', value)}
                  className={!validateListButtonName(nodeSettings.listButtonName) ? 'border-red-500' : ''}
                />
                {!validateListButtonName(nodeSettings.listButtonName) && (
                  <p className="text-xs text-red-500">List button name exceeds WhatsApp limit</p>
                )}
              </div>
            )}

            <div className="space-y-3">
              {Array.from({ length: nodeSettings.activeButtons }).map((_, index) => {
                const buttonNum = index + 1;
                return (
                  <div key={buttonNum} className="relative">
                    <div className="flex items-start gap-2">
                      <div className="flex-1">
                        <div className="flex justify-between items-center">
                          <Label htmlFor={`${id}-button-${buttonNum}`}>Button {buttonNum}</Label>
                          <span className={`text-xs ${getCharacterCount(String(nodeSettings[`button${buttonNum}`])) > getButtonCharLimit() ? 'text-red-500' : 'text-gray-500'}`}>
                            {getCharacterCount(String(nodeSettings[`button${buttonNum}`]))}/{getButtonCharLimit()}
                          </span>
                        </div>
                        <VariableInput
                          id={`${id}-button-${buttonNum}`}
                          placeholder={`Enter button ${buttonNum} text`}
                          value={String(nodeSettings[`button${buttonNum}`])}
                          onChange={(value) => updateNodeData(`button${buttonNum}`, value)}
                          className={!validateButtonText(String(nodeSettings[`button${buttonNum}`])) ? 'border-red-500' : ''}
                        />
                        {!validateButtonText(String(nodeSettings[`button${buttonNum}`])) && (
                          <p className="text-xs mt-3 text-red-500">
                            Max ({getButtonCharLimit()} char)
                          </p>
                        )}
                      </div>
                      {nodeSettings.activeButtons > 1 && (
                        <Button
                          variant="ghost"
                          size="icon"
                          className="mt-6"
                          onClick={() => removeButton(buttonNum)}
                        >
                          <X className="h-4 w-4" />
                        </Button>
                      )}
                    </div>
                    <Handle
                      type="source"
                      position={Position.Right}
                      id={`button-${buttonNum}`}
                      className="!bg-gray-400 !w-3 !h-3 !min-w-[12px] !min-h-[12px] !border-2 !border-white"
                      style={{
                        right: '-20px',
                        top: '50%',
                        transform: 'translateY(-50%)',
                        zIndex: 50
                      }}
                      isConnectable={isConnectable}
                    />
                  </div>
                );
              })}
              
              {nodeSettings.activeButtons < getButtonLimit() && (
                <Button
                  variant="outline"
                  className="w-full"
                  onClick={addButton}
                >
                  <Plus className="h-4 w-4 mr-2" />
                  Add Button
                </Button>
              )}
              {nodeSettings.activeButtons >= getButtonLimit() && (
                <div className="text-xs text-gray-500 text-center py-2">
                  Maximum {getButtonLimit()} buttons allowed for {isListMode ? 'List Message' : 'Quick Reply'}
                </div>
              )}

              <div className="mt-3 pt-2 border-t border-green-200 flex items-center justify-end gap-2">
                <div className="flex flex-col items-end">
                  <span className="text-xs text-gray-500">Else exit</span>
                  <span className="text-[10px] text-gray-400">Triggered when user reply is not known option</span>
                </div>
                <Handle
                  type="source"
                  position={Position.Right}
                  id="else"
                  className="!bg-gray-400 !w-3 !h-3 !min-w-[12px] !min-h-[12px] !border-2 !border-white"
                  style={{ 
                    position: 'relative',
                    right: '-8px',
                    transform: 'translateY(0)',
                    display: 'inline-block'
                  }}
                  isConnectable={isConnectable}
                />
              </div>
            </div>
          </div>
        </div>
      </ContextMenuTrigger>
      <ContextMenuContent>
        <ContextMenuItem
          className="text-red-600 focus:text-red-600 focus:bg-red-100"
          onClick={() => deleteNode(id)}
        >
          <Trash2 className="mr-2 h-4 w-4" />
          Delete
        </ContextMenuItem>
      </ContextMenuContent>
    </ContextMenu>
  );
};

export default QuickRepliesNode;
