<?php

namespace Modules\Flowmaker\Models\Nodes;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Modules\Flowmaker\Models\Contact;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Mail\Mailables\Attachment;

class SendEmail extends Node
{
    public function process($message, $data)
    {
        Log::info('🔄 Processing SendEmail node', [
            'node_id' => $this->id ?? 'unknown',
            'flow_id' => $this->flow_id ?? 'unknown'
        ]);

        try {
            // Load settings
            $nodeData = $this->getDataAsArray();
            $settings = $nodeData['settings']['sendEmail'] ?? [];
            Log::debug('📋 SendEmail settings', ['settings' => $settings]);

            // Resolve contact
            $contactId = is_object($data) ? ($data->contact_id ?? null) : ($data['contact_id'] ?? null);
            $contact = $contactId ? Contact::find($contactId) : null;
            if (!$contact) {
                Log::error('❌ Contact not found', ['contactId' => $contactId]);
                return ['success' => false];
            }

            // Extract settings
            $to = $settings['to'] ?? '';
            $cc = $settings['cc'] ?? '';
            $bcc = $settings['bcc'] ?? '';
            $subject = $settings['subject'] ?? '';
            $body = $settings['body'] ?? '';
            $isHtml = $settings['isHtml'] ?? true;
            $fromEmail = $settings['fromEmail'] ?? '';
            $fromName = $settings['fromName'] ?? '';
            $responseVar = trim($settings['responseVar'] ?? '');
            $attachments = $settings['attachments'] ?? [];
            $smtpConfig = $settings['smtpConfig'] ?? [];

            // Interpolate variables
            $to = $contact->changeVariables($to, $this->flow_id);
            $cc = $contact->changeVariables($cc, $this->flow_id);
            $bcc = $contact->changeVariables($bcc, $this->flow_id);
            $subject = $contact->changeVariables($subject, $this->flow_id);
            $body = $contact->changeVariables($body, $this->flow_id);
            $fromEmail = $contact->changeVariables($fromEmail, $this->flow_id);
            $fromName = $contact->changeVariables($fromName, $this->flow_id);

            if (empty($to)) {
                Log::error('❌ Recipient email is empty');
                return ['success' => false, 'error' => 'Recipient email is required'];
            }

            // Create custom mailable
            $mailable = new class($subject, $body, $isHtml, $fromEmail, $fromName, $attachments) extends Mailable {
                public $subject;
                public $body;
                public $isHtml;
                public $fromEmail;
                public $fromName;
                public $attachments;

                public function __construct($subject, $body, $isHtml, $fromEmail, $fromName, $attachments)
                {
                    $this->subject = $subject;
                    $this->body = $body;
                    $this->isHtml = $isHtml;
                    $this->fromEmail = $fromEmail;
                    $this->fromName = $fromName;
                    $this->attachments = $attachments;
                }

                public function envelope()
                {
                    $envelope = new Envelope();
                    $envelope->subject($this->subject);
                    
                    if (!empty($this->fromEmail)) {
                        $envelope->from($this->fromEmail, $this->fromName);
                    }
                    
                    return $envelope;
                }

                public function content()
                {
                    return new Content(
                        $this->isHtml ? 'html' : 'text',
                        'flowmaker::emails.dynamic',
                        ['body' => $this->body]
                    );
                }

                public function attachments()
                {
                    $attachments = [];
                    foreach ($this->attachments as $attachment) {
                        if (isset($attachment['path']) && file_exists($attachment['path'])) {
                            $attachments[] = Attachment::fromPath($attachment['path'])
                                ->as($attachment['name'] ?? basename($attachment['path']));
                        }
                    }
                    return $attachments;
                }
            };

            // Configure SMTP if custom settings provided
            if (!empty($smtpConfig) && $smtpConfig['useCustom']) {
                $this->configureCustomSMTP($smtpConfig);
            }

            // Send email
            $mailer = Mail::mailer();
            
            if (!empty($cc)) {
                $ccEmails = array_map('trim', explode(',', $cc));
                $mailable->cc($ccEmails);
            }
            
            if (!empty($bcc)) {
                $bccEmails = array_map('trim', explode(',', $bcc));
                $mailable->bcc($bccEmails);
            }

            $result = $mailer->to($to)->send($mailable);

            // Store response in contact state
            if ($responseVar !== '') {
                $response = [
                    'success' => true,
                    'sent_at' => now()->toISOString(),
                    'to' => $to,
                    'subject' => $subject,
                    'cc' => $cc,
                    'bcc' => $bcc
                ];
                $contact->setContactState($this->flow_id, $responseVar, json_encode($response));
                Log::info('✅ Stored email response in contact state', ['variable' => $responseVar]);
            }

            // Continue to next node
            $nextNode = $this->getNextNodeId();
            if ($nextNode) {
                $nextNode->process($message, $data);
            }

            return ['success' => true, 'result' => $result];

        } catch (\Exception $e) {
            Log::error('❌ SendEmail node error', ['error' => $e->getMessage()]);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    private function configureCustomSMTP($smtpConfig)
    {
        config([
            'mail.mailers.custom' => [
                'transport' => 'smtp',
                'host' => $smtpConfig['host'] ?? 'smtp.gmail.com',
                'port' => $smtpConfig['port'] ?? 587,
                'encryption' => $smtpConfig['encryption'] ?? 'tls',
                'username' => $smtpConfig['username'] ?? '',
                'password' => $smtpConfig['password'] ?? '',
                'timeout' => $smtpConfig['timeout'] ?? 30,
            ]
        ]);
    }

    protected function getNextNodeId($data = null)
    {
        if (!empty($this->outgoingEdges)) {
            return $this->outgoingEdges[0]->getTarget();
        }
        return null;
    }
}